/** 
 * @file    kPxBlock.x.h
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PX_BLOCK_X_H
#define K_FIRESYNC_PX_BLOCK_X_H

#include <kFireSync/Pipe/kPxPort.h>

typedef struct kPxBlockClass
{
    kObjectClass base; 
    kObject owner;                      //Optional reference to owner (can be used in pipe execution engine). 
    kPxEnviron pipeEnviron;             //Provides services from execution environment.
    kAlloc msgAlloc;                    //Message allocator. 
    kCallback sendFx;                   //Message send handler. 
    kCallback dropFx;                   //Message drop handler. 
    kText32 name;                       //Instance name. 
    k32u blockId;                       //Block instance identifier. 
    k32u nodeId;                        //Node identifier. 
    kXml settings;                      //XML settings. 
    kBool isGrouped;                    //Is this block assigned to a processing group?
    k32u groupId;                       //Optional processing group id. 
    kArrayList ports;                   //List of defined ports -- kArrayList<kPxPort>.
    kBool hasQueuedInputs;              //Are message input queues enabled for this block?
    kMsgSource msgSource;               //Input message source (valid during Process call). 
    kBool isStarted;                    //Has the block been successfully started?
    kBool isPaused;                     //Has the block been successfully paused?
    kBool hasProcessingError;           //Has a processing error occurred since latest Start.
    kProfileProbe receiveProbe;         //Timer probe for receive operation.
    kHealthProbe dropProbe;             //Health probe for message drops. 
} kPxBlockClass; 

typedef struct kPxBlockVTable
{
    kObjectVTable base; 
    
    const kChar* (kCall* VDescription)();
    kStatus (kCall* VInit)(kPxBlock block, kType type, const kChar* name, k32u nodeId, k32u blockId, kPxEnviron pipeEnviron, kAlloc alloc);
    kStatus (kCall* VOnSetup)(kPxBlock block);
    kStatus (kCall* VOnStart)(kPxBlock block);
    kStatus (kCall* VOnEngage)(kPxBlock block);
    kStatus (kCall* VOnStop)(kPxBlock block);
    kStatus (kCall* VOnPause)(kPxBlock block);
    kStatus (kCall* VOnResume)(kPxBlock block);
    kStatus (kCall* VOnReceive)(kPxBlock block, kPxPort port, kMsgSet msg);

} kPxBlockVTable; 

kDeclareVirtualClassEx(kFs, kPxBlock, kObject)
        

kFsFx(kStatus) kPxBlock_VInit(kPxBlock block, kType type, const kChar* name, k32u nodeId, k32u blockId, kPxEnviron pipeEnviron, kAlloc alloc);

kFsFx(kStatus) kPxBlock_VRelease(kPxBlock block);

kFsFx(const kChar*) kPxBlock_VDescription();

kFsFx(kStatus) kPxBlock_VOnSetup(kPxBlock block);
kFsFx(kStatus) kPxBlock_VOnStart(kPxBlock block);
kFsFx(kStatus) kPxBlock_VOnEngage(kPxBlock block);
kFsFx(kStatus) kPxBlock_VOnStop(kPxBlock block);
kFsFx(kStatus) kPxBlock_VOnPause(kPxBlock block);
kFsFx(kStatus) kPxBlock_VOnResume(kPxBlock block);
kFsFx(kStatus) kPxBlock_VOnReceive(kPxBlock block, kPxPort port, kMsgSet msg);

kFsFx(kStatus) kPxBlock_ClearPorts(kPxBlock block);
kFsFx(kStatus) kPxBlock_ClearQueuedMessages(kPxBlock block);

kFsFx(kStatus) kPxBlock_EnableGroup(kPxBlock block, kBool enabled);
kFsFx(kStatus) kPxBlock_SetGroupId(kPxBlock block, k32u groupId);

kFsFx(kStatus) kPxBlock_ZeroContext(kPxBlock block); 

kFsFx(kStatus) kPxBlock_ClearStats(kPxBlock block); 
kFsFx(kStatus) kPxBlock_EnsureCompatibleMemory(kPxBlock block, kPxPort port, kMsgInfo input, kMsgInfo* compatibleInput);

kFsFx(kStatus) kPxBlock_SetOwner(kPxBlock block, kObject owner);

kFsFx(k32u) kPxBlock_NodeId(kPxBlock block);
kFsFx(kXmlItem) kPxBlock_SettingsRoot(kPxBlock block);

kInlineFx(kObject) kPxBlock_Owner(kPxBlock block)
{
    kObj(kPxBlock, block); 

    return obj->owner;
}

/* 
* Deprecated (Stage 1): not recommended for further use, but not yet announced via kDeprecate
*/

//[Deprecated] No longer used/required by pipe framework.
kFsFx(kStatus) kPxBlock_OnPause(kPxBlock block);

//[Deprecated] No longer used/required by pipe framework.
kFsFx(kStatus) kPxBlock_OnResume(kPxBlock block);

//[Deprecated] No longer used/required by pipe framework.
kFsFx(kStatus) kPxBlock_Pause(kPxBlock block);

//[Deprecated] No longer used/required by pipe framework.
kFsFx(kStatus) kPxBlock_Resume(kPxBlock block);

#endif
