/** 
 * @file    kDhcpClient.x.h
 *
 * @internal
 * Copyright (C) 2019-2022 by LMI Technologies Inc.
 */
#ifndef K_FIRESYNC_DHCP_CLIENT_X_H
#define K_FIRESYNC_DHCP_CLIENT_X_H

#include <kFireSync/Net/kPacketUtils.h>

/* DHCP states */
#define xkDHCP_CLIENT_STATE_UNKNOWN      (0)   // Invalid state.
#define xkDHCP_CLIENT_STATE_INIT         (1)   // Client begins the process of acquiring a lease. Also returns here when negotiation fails.
#define xkDHCP_CLIENT_STATE_SELECTING    (2)   // Client is waiting to receive DHCPOFFER message.
#define xkDHCP_CLIENT_STATE_REQUESTING   (3)   // Client is waiting for the server to answer request.
#define xkDHCP_CLIENT_STATE_BOUND        (4)   // Client has lease and is in normal operation state.
#define xkDHCP_CLIENT_STATE_RENEWING     (5)   // Client is trying to renew lease from the original server.
#define xkDHCP_CLIENT_STATE_REBINDING    (6)   // Client is trying to renew lease from any server.

/* Timeouts */
#define xkDHCP_CLIENT_DISCOVER_INIT_TIMEOUT             (2000000)       // Initial retry timeout for discover messages.
#define xkDHCP_CLIENT_DISCOVER_MAX_TIMEOUT              (60000000)      // Maximal retry timeout for discover messages.
#define xkDHCP_CLIENT_REQUEST_INIT_TIMEOUT              (10000000)      // Initial retry timeout for request messages.
#define xkDHCP_CLIENT_REQUEST_MAX_TIMEOUT               (60000000)      // Maximal retry timeout for request messages.

#define xkDHCP_CLIENT_RECV_TIMEOUT                      (100000)        // Maximal timeout for receiving messages.
#define xkDHCP_CLIENT_RECV_REPLY_TIMEOUT                (2000000)       // Maximal timeout for DHCP replies.

#define xkDHCP_CLIENT_DISCOVER_MAX_INIT_DELAY           (2000000)       // Maximal delay during init state.

#define xkDHCP_CLIENT_REQUEST_MAX_RT_COUNT              (4)             // Retransmit the DHCPREQUEST message up to four times (RfC proposal).
#define xkDHCP_CLIENT_REQUEST_MIN_DELAY                 (10000000)      // Minimum delay when sending DHCPREQUEST messages in RENEWING and REBINDING states.

#define xkDHCP_CLIENT_INFINITE                          (0xFFFFFFFF)    // Infinite lease time.

#define xkDHCP_CLIENT_DEF_OPTION_PARAMETER              xkPACKET_UTILS_DHCP_OPT_SUBNET_MASK, xkPACKET_UTILS_DHCP_OPT_ROUTER, xkPACKET_UTILS_DHCP_OPT_LEASE_TIME, \
                                                        xkPACKET_UTILS_DHCP_OPT_RENEWAL_TIME, xkPACKET_UTILS_DHCP_OPT_REBINDING_TIME

#define xkDHCP_CLIENT_LOG                               (kFALSE)        // Set to kTRUE to enable logging.
#define xkDHCP_CLIENT_HEARTBEAT                         (500000)        // Heartbeat (2Hz).

#define xkDHCP_CLIENT_MAX_PACKET_COUNT                  (20)            // Maximal number of queued DHCP packets.

#define xkDHCP_CLIENT_NEGOTIATION_TIMEOUT               (40000000)      // Initial retry timeout for discover messages.

kDeclareEnumEx(kFs, kDhcpEvent, kValue)

typedef kStatus (kCall* kDhcpParseFx)(kDhcpClient client, kArray1 data); 

typedef struct kDhcpClientClass
{   
    kObjectClass base; 

    kText256 adapterName;           // Name of the NIC.
    kThread recvThread;             // Receive thread.
    kAtomic32s shouldQuit;          // Thread quit flag;
    kPeriodic heartBeat;            // DHCP heartbeat handler.
    kPacketSocket sockt;            // Socket used for sending/receiving raw packets.
    kArp arp;                       // Arp object used for probing/announcing.
    kMsgQueue packetQueue;          // kMsgQueue<kArray1<kByte>>
    kArray1 packet;                 // kArray1<kByte>, used to temporary store ethernet frames.

    kAtomic32s state;               // Current DHCP state. 
    k64u stateTimeStamp;            // Time stamp of latest action (sending messages) in current state.
    k64u stateTimeOut;              // Time out of current state.
    kSize stateRetransmitCount;     // Transmit count in current state.
    k64u retransmitTimeout;         // Current retransmit timeout.

    k32u transactionId;             // DHCP Transaction Id.
    k64u configStartTime;           // Start of acquisition or renewal.
    k64u leaseStartTime;            // Start of the lease.
    k64u leaseTime;                 // Lease time in microseconds.
    k64u t1;                        // Renewal time in microseconds.
    k64u t2;                        // Rebinding time in microseconds.

    kEvent statusEvent;             // IP config status event.
    kPeriodic negotiationTimeout;   // Periodic to notify in case of timeout.

    kMacAddress macAddress;         // Our own MAC address.
    kMacAddress serverMac;          // MAC address of the server.
    
    kIpAddress address;             // The IP we got from the server.
    k32u prefix;                    // The prefix we got from the server.
    kIpAddress gateway;             // The gateway IP we got from the server.
    kIpAddress serverAddress;       // The server we got the IP from.
    
} kDhcpClientClass;

kDeclareClassEx(kFs, kDhcpClient, kObject)

/* 
* Private methods. 
*/

kFsFx(kStatus) xkDhcpClient_Init(kDhcpClient client, kType type, kAlloc alloc); 
kFsFx(kStatus) xkDhcpClient_VRelease(kDhcpClient client);

kFsFx(kStatus) xkDhcpClient_HeartBeat(kDhcpClient client, kPeriodic timer);
kFsFx(kStatus) xkDhcpClient_RecvThread(kDhcpClient client);
kFsFx(kStatus) xkDhcpClient_StartReceiveThread(kDhcpClient client);
kFsFx(kStatus) xkDhcpClient_StopReceiveThread(kDhcpClient client);

kFsFx(kStatus) xkDhcpClient_QueuePacket(kDhcpClient client, const kByte* buffer, kSize bufferSize);

kFsFx(kStatus) xkDhcpClient_SendDhcpDiscover(kDhcpClient client);
kFsFx(kStatus) xkDhcpClient_SendDhcpRequest(kDhcpClient client);
kFsFx(kStatus) xkDhcpClient_SendDhcpDecline(kDhcpClient client);

kFsFx(kStatus) xkDhcpClient_ReleaseDhcpLease(kDhcpClient client);

kFsFx(kBool) xkDhcpClient_IsDhcpReply(kDhcpClient client, kByte* frame, kSize size);
kFsFx(kStatus) xkDhcpClient_FindReply(kDhcpClient client, kDhcpParseFx parseFx);
kFsFx(kStatus) xkDhcpClient_ParseOffer(kDhcpClient client, kArray1 data);
kFsFx(kStatus) xkDhcpClient_ParseAck(kDhcpClient client, kArray1 data);
kFsFx(kStatus) xkDhcpClient_ParseNak(kDhcpClient client, kArray1 data);

kFsFx(kStatus) xkDhcpClient_SendFrame(kDhcpClient client, kByte* frame, kSize size, kIpAddress sourceIp, kIpAddress destIp, kMacAddress macAddress, kMacAddress destMac, kSize dataLength);

kFsFx(kStatus) xkDhcpClient_ProcessStateInit(kDhcpClient client);
kFsFx(kStatus) xkDhcpClient_ProcessStateSelecting(kDhcpClient client);
kFsFx(kStatus) xkDhcpClient_ProcessStateRequesting(kDhcpClient client);
kFsFx(kStatus) xkDhcpClient_ProcessStateBound(kDhcpClient client);
kFsFx(kStatus) xkDhcpClient_ProcessStateRenewing(kDhcpClient client);
kFsFx(kStatus) xkDhcpClient_ProcessStateRebinding(kDhcpClient client);
kFsFx(kStatus) xkDhcpClient_ChangeState(kDhcpClient client, k32s state, kSize delay);

kFsFx(kStatus) xkDhcpClient_TimeoutElapsed(kDhcpClient client, kPeriodic timer); 

kFsFx(kStatus) xkDhcpClient_Notify(kDhcpClient client, kDhcpEvent eventType);

kFsFx(kStatus) xkDhcpClient_LogState(kDhcpClient client);

k16u xkDhcpClient_Time(k64u t);

template<typename T>
T xkDhcpClient_randRanged(T minValue, T maxValue)
{
    if (minValue >= maxValue)
    {
        return minValue;
    }

    return kRandom64u() % (maxValue + 1 - minValue) + minValue;
}

#endif
