/** 
 * @file    kFireSync/Health/kProfileProbe.h
 * @brief   Declares the kProfileProbe class and related types. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PROFILE_PROBE_H
#define K_FIRESYNC_PROFILE_PROBE_H

#include <kFireSync/kNodeDef.h>
#include <kFireSync/kFsLib.h>

/**
 * @struct   kProfileContext
 * @extends  kValue
 * @ingroup  kFireSync-Health
 * @brief    Context information for a code profiling measurement. 
 * @see      kProfileProbe, kProfileProbe_StartEx
 */
typedef k64u kProfileContext; 

#include <kFireSync/Health/kProfileProbe.x.h>

/**
 * @class   kProfileProbe
 * @ingroup kFireSync-Health
 * @brief   Represents a code profiling probe.
 * 
 * Profiling probes are associated with a health collection provider. A default health 
 * collection provider may be installed by the application environment.  If not, or if 
 * a more specific collection provider is desired, the kProfileProbe_ConstructEx constructor 
 * can be used to construct a probe and attach it to the specified provider. 
 * 
 * Code profiling probes are intended to efficiently measure the elapsed time of relatively 
 * brief operations. As such, profiling timers may overflow and produce incorrect results if used 
 * to measure operations that require longer than a few seconds.  If a long duration needs 
 * to be meausred, consider using kTimer and kHealthProbe instead of kProfileProbe.
 * 
 * Code profiling is effectively disabled unless the K_PROFILE prepocessor 
 * flag is defined. 
 */
//typedef kObject kProfileProbe;            --forward-declared in kFsDef.x.h  

/** 
 * Constructs a kProfileProbe object attached to the default health provider.
 *
 * @public              @memberof kProfileProbe
 * @param   probe       Destination for the constructed object handle. 
 * @param   name        Descriptive label (e.g. "Hw/Camera0/ExposureCount").
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kInlineFx(kStatus) kProfileProbe_Construct(kProfileProbe* probe, const kChar* name, kAlloc allocator)
{
#   if defined(K_PROFILE)
    {
        return xkProfileProbe_ConstructImpl(probe, kFsLib_HealthProvider(), name, allocator);
    }
#   else
    {
        return xkProfileProbe_ConstructImpl(probe, kNULL, name, allocator);
    }
#   endif
}

/** 
 * Constructs a kProfileProbe object attached to the specified health provider.
 *
 * @public                  @memberof kProfileProbe
 * @param   provider        Health provider (or kNULL for default). 
 * @param   probe           Destination for the constructed object handle. 
 * @param   name            Descriptive label (e.g. "Hw/Camera0/ExposureCount").
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kInlineFx(kStatus) kProfileProbe_ConstructEx(kProfileProbe* probe, kHealth provider, const kChar* name, kAlloc allocator)
{
#   if defined(K_PROFILE)
    {
        kHealth sanitizedProvider = !kIsNull(provider) ? provider : kFsLib_HealthProvider(); 

        return xkProfileProbe_ConstructImpl(probe, sanitizedProvider, name, allocator);
    }
#   else
    {
        return xkProfileProbe_ConstructImpl(probe, kNULL, name, allocator);
    }
#   endif
}

/** 
 * Begins a code profiling measurement. 
 *
 * @public              @memberof kProfileProbe
 * @param   probe       Probe object.  
 */
kInlineFx(void) kProfileProbe_Start(kProfileProbe probe)
{
#   if defined(K_PROFILE)
    {
        xkProfileProbe_VTable(probe)->VStart(probe);
    }
#   endif
}

/** 
 * Ends a code profiling measurement. 
 *
 * @public              @memberof kProfileProbe
 * @param   probe       Probe object.  
 */
kInlineFx(void) kProfileProbe_Stop(kProfileProbe probe)
{
#   if defined(K_PROFILE)
    {
        xkProfileProbe_VTable(probe)->VStop(probe); 
    }
#   endif
}

/** 
 * Begins a thread-safe code profiling measurement. 
 *
 * kProfileProbe_StartEx ad kProfileProbe_StopEx should be used instead of kProfileProbe_Start 
 * and kProfileProbe_Stop if the code under test may be executed simultaneously by multiple 
 * threads. 
 * 
 * @public              @memberof kProfileProbe
 * @param   probe       Probe object.  
 * @param   context     Measurement context information.   
 */
kInlineFx(void) kProfileProbe_StartEx(kProfileProbe probe, kProfileContext* context)
{
#   if defined(K_PROFILE)
    {
        xkProfileProbe_VTable(probe)->VStartEx(probe, context);
    }
#   endif
}

/** 
 * Ends a thread-safe code profiling measurement. 
 *
 * @public              @memberof kProfileProbe
 * @param   probe       Probe object.  
 * @param   context     Measurement context information.    
 */
kInlineFx(void) kProfileProbe_StopEx(kProfileProbe probe, kProfileContext* context)
{
#   if defined(K_PROFILE)
    {
        xkProfileProbe_VTable(probe)->VStopEx(probe, context);
    }
#   endif
}

/** 
 * Reports the name of the profile probe. 
 *
 * @public              @memberof kProfileProbe
 * @param   probe       Probe object.
 * @return              Name of the profile probe. 
 */
kInlineFx(const kChar*) kProfileProbe_Name(kProfileProbe probe)
{
    kObj(kProfileProbe, probe);

    return obj->name;
}

#endif


