/** 
 * @file    kHxProjector.h
 * @brief   Declares the kHxProjector class.
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_FIRESYNC_HX_PROJECTOR_H
#define K_FIRESYNC_HX_PROJECTOR_H

#include <kFireSync/kNodeDef.h>
#include <kFireSync/Client/Projector/kProjectorInfo.h>
#include <kFireSync/Hardware/kHxProjector.x.h>

/**
 * @class       kHxProjector
 * @extends     kObject
 * @ingroup     kFireSync-Hardware
 * @internal
 * @brief       Abstract base class for hardware projector classes.
 */
//typedef kObject kHxProjector;      --forward-declared in kFsDef.x.h

kInlineFx(kHxNode) kHxProjector_Node(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->node;
}

kInlineFx(kSize) kHxProjector_Index(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->index;
}

kInlineFx(kProjectorInfo) kHxProjector_Info(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->info;
}

kInlineFx(kProjectorModel) kHxProjector_Model(kHxProjector projector)
{
    kObj(kHxProjector, projector); 
    
    return kProjectorInfo_Model(obj->info); 
}

kInlineFx(k64f) kHxProjector_PixelClockFrequency(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->pixelClockFrequency;
}

kInlineFx(kBool) kHxProjector_IsConnected(kHxProjector projector)
{
    return xkHxProjector_VTable(projector)->VIsConnected(projector);
}

kInlineFx(kStatus) kHxProjector_Enable(kHxProjector projector, kBool enabled)
{
    kObj(kHxProjector, projector); 

    obj->enabled = enabled; 

    return kOK; 
}

kInlineFx(kBool) kHxProjector_IsEnabled(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->enabled; 
}

kInlineFx(kStatus) kHxProjector_SetControl(kHxProjector projector, kProjectorControl type)
{
    kObj(kHxProjector, projector); 

    obj->controlType = type; 

    return kOK; 
}

kInlineFx(kProjectorControl) kHxProjector_Control(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->controlType; 
}

kInlineFx(kStatus) kHxProjector_SetControlId(kHxProjector projector, kSize id)
{
    kObj(kHxProjector, projector); 

    obj->controlId = id; 

    return kOK; 
}

kInlineFx(kSize) kHxProjector_ControlId(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->controlId; 
}

kInlineFx(kStatus) kHxProjector_EnableDynamicFrameRate(kHxProjector projector, kBool enable)
{
    kObj(kHxProjector, projector); 

    obj->dynamicFrameRate = enable; 

    return kOK; 
}

kInlineFx(kBool) kHxProjector_IsDynamicFrameRateEnabled(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->dynamicFrameRate; 
}

kInlineFx(kStatus) kHxProjector_EnableLightOutput(kHxProjector projector, kBool enable)
{
    kObj(kHxProjector, projector); 

    obj->lightOutput = enable; 

    return kOK; 
}

kInlineFx(kBool) kHxProjector_LightOutputEnabled(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->lightOutput; 
}

kInlineFx(kStatus) kHxProjector_EnableAutoShutdown(kHxProjector projector, kBool enable)
{
    kObj(kHxProjector, projector);

    obj->autoShutdown = enable;

    return kOK;
}

kInlineFx(kBool) kHxProjector_AutoShutdownEnabled(kHxProjector projector)
{
    kObj(kHxProjector, projector);

    return obj->autoShutdown;
}

kInlineFx(kStatus) kHxProjector_SetPatternCount(kHxProjector projector, kSize patternCount)
{
    kObj(kHxProjector, projector); 

    obj->patternCount = patternCount; 

    return kOK; 
}

kInlineFx(kSize) kHxProjector_PatternCount(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->patternCount; 
}

kInlineFx(kStatus) kHxProjector_SetBitDepth(kHxProjector projector, kSize bitDepth)
{
    kObj(kHxProjector, projector); 

    obj->bitDepth = bitDepth; 

    return kOK; 
}

kInlineFx(kSize) kHxProjector_BitDepth(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->bitDepth; 
}

kInlineFx(kStatus) kHxProjector_SetPatternMode(kHxProjector projector, kProjectorPatternMode mode)
{
    kObj(kHxProjector, projector); 

    obj->patternMode = mode; 

    return kOK; 
}

kInlineFx(kProjectorPatternMode) kHxProjector_PatternMode(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return obj->patternMode; 
}

kInlineFx(kStatus) kHxProjector_SetPalette(kHxProjector projector, kSize index, k32u color)
{
    kObj(kHxProjector, projector); 

    kCheckArgs(index < kPROJECTOR_PALETTE_CAPACITY);

    obj->palette[index] = color; 

    return kOK; 
}

kInlineFx(k32u) kHxProjector_Palette(kHxProjector projector, kSize index)
{
    kObj(kHxProjector, projector); 

    if (index >= kPROJECTOR_PALETTE_CAPACITY)
    {
        return 0;
    }

    return obj->palette[index];  
}

kInlineFx(k32u) kHxProjector_MinIntensity(kHxProjector projector)
{
    return xkHxProjector_VTable(projector)->VMinIntensity(projector);
}

kInlineFx(k32u) kHxProjector_MaxIntensity(kHxProjector projector)
{
    return xkHxProjector_VTable(projector)->VMaxIntensity(projector);
}

kInlineFx(kStatus) kHxProjector_LoadPatterns(kHxProjector projector, kArray2 patterns)
{
    return xkHxProjector_VTable(projector)->VLoadPatterns(projector, patterns);
}

kInlineFx(kStatus) kHxProjector_LoadRowGains(kHxProjector projector, kArray1 gains)
{
    return xkHxProjector_VTable(projector)->VLoadRowGains(projector, gains);
}

kInlineFx(kStatus) kHxProjector_LoadColGains(kHxProjector projector, kArray1 gains)
{
    return xkHxProjector_VTable(projector)->VLoadColGains(projector, gains);
}

kInlineFx(kStatus) kHxProjector_SetStateCount(kHxProjector projector, kSize count)
{
    return xkHxProjector_VTable(projector)->VSetStateCount(projector, count);
}

kInlineFx(kSize) kHxProjector_StateCount(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return kArrayList_Count(obj->states);
}

kInlineFx(kHxProjectorState) kHxProjector_StateAt(kHxProjector projector, kSize index)
{
    kObj(kHxProjector, projector); 

    return kArrayList_AsT(obj->states, index, kHxProjectorState);
}

kInlineFx(kStatus) kHxProjector_ClearExtensions(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return kArrayList_Purge(obj->extensions);
}

 //ownership of 'extension' configuration argument not transferred; 
kInlineFx(kStatus) kHxProjector_AddExtension(kHxProjector projector, kProjectorExt extension)
{
    return xkHxProjector_VTable(projector)->VAddExtension(projector, extension);
}

kInlineFx(kSize) kHxProjector_ExtensionCount(kHxProjector projector)
{
    kObj(kHxProjector, projector); 

    return kArrayList_Count(obj->extensions);
}

//produces object that must be destroyed by caller
kFsFx(kStatus) kHxProjector_ExtensionAt(kHxProjector projector, kSize index, kProjectorExt* extension, kAlloc allocator);

kInlineFx(kStatus) kHxProjector_Stats(kHxProjector projector, kProjectorStats* stats)
{
    return xkHxProjector_VTable(projector)->VStats(projector, stats);
}

kInlineFx(kStatus) kHxProjector_Clear(kHxProjector projector)
{
    return xkHxProjector_VTable(projector)->VClear(projector);
}

#endif
