/** 
 * @file    kHxLight.h
 * @brief   Declares the kHxLight class.
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_FIRESYNC_HX_LIGHT_H
#define K_FIRESYNC_HX_LIGHT_H

#include <kFireSync/kNodeDef.h>
#include <kFireSync/Utils/kLdModel.h>
#include <kFireSync/Hardware/kHxLight.x.h>

/**
 * @class       kHxLight
 * @extends     kObject
 * @ingroup     kFireSync-Hardware
 * @internal
 * @brief       Abstract base class for hardware light classes.
 */
//typedef kObject kHxLight;      --forward-declared in kFsDef.x.h

kInlineFx(kHxNode) kHxLight_Node(kHxLight light)
{
    kObj(kHxLight, light); 

    return obj->node;
}

kInlineFx(kSize) kHxLight_Index(kHxLight light)
{
    kObj(kHxLight, light); 

    return obj->index;
}

kInlineFx(kLightModel) kHxLight_Model(kHxLight light)
{
    kObj(kHxLight, light); 

    return kLdModel_LightModel(obj->info); 
}

kInlineFx(kLightDiodeModel) kHxLight_DiodeModel(kHxLight light)
{
    kObj(kHxLight, light); 

    return kLdModel_DiodeModel(obj->info); 
}

kInlineFx(kLightCapability) kHxLight_Capabilities(kHxLight light)
{
    kObj(kHxLight, light); 

    return obj->capabilities;
}

kInlineFx(kStatus) kHxLight_Enable(kHxLight light, kBool enabled)
{
    kObj(kHxLight, light); 

    obj->enabled = enabled; 

    return kOK; 
}

kInlineFx(kBool) kHxLight_IsEnabled(kHxLight light)
{
    kObj(kHxLight, light); 

    return obj->enabled; 
}

kInlineFx(kStatus) kHxLight_SetControl(kHxLight light, kLightControl type)
{
    kObj(kHxLight, light); 

    obj->controlType = type; 

    return kOK; 
}

kInlineFx(kLightControl) kHxLight_Control(kHxLight light)
{
    kObj(kHxLight, light); 

    return obj->controlType; 
}

kInlineFx(kStatus) kHxLight_SetControlId(kHxLight light, kSize id)
{
    kObj(kHxLight, light); 

    obj->controlId = id; 

    return kOK; 
}

kInlineFx(kSize) kHxLight_ControlId(kHxLight light)
{
    kObj(kHxLight, light); 

    return obj->controlId; 
}

kInlineFx(kStatus) kHxLight_SetIntensity(kHxLight light, k64f intensity)
{
    return xkHxLight_VTable(light)->VSetIntensity(light, intensity);
}

kInlineFx(k64f) kHxLight_Intensity(kHxLight light)
{
    kObj(kHxLight, light); 

    return obj->intensity; 
}

kInlineFx(k64f) kHxLight_MinIntensity(kHxLight light)
{
    kObj(kHxLight, light); 

    return obj->minIntensity; 
}

kInlineFx(kStatus) kHxLight_SetMask(kHxLight light, kLightMask mask)
{
    kObj(kHxLight, light); 

    obj->mask = mask; 

    return kOK; 
}

kInlineFx(kLightMask) kHxLight_Mask(kHxLight light)
{
    kObj(kHxLight, light); 

    return obj->mask; 
}

kInlineFx(kStatus) kHxLight_SetMaskId(kHxLight light, k32u id)
{
    kObj(kHxLight, light); 

    obj->maskId = id; 

    return kOK; 
}

kInlineFx(k32u) kHxLight_MaskId(kHxLight light)
{
    kObj(kHxLight, light); 

    return obj->maskId;     
}

kInlineFx(kStatus) kHxLight_SetMaskPolarity(kHxLight light, kBool isActiveHigh)
{
    kObj(kHxLight, light); 

    obj->maskPolarity = isActiveHigh; 

    return kOK; 
}

kInlineFx(kBool) kHxLight_MaskPolarity(kHxLight light)
{
    kObj(kHxLight, light); 

    return obj->maskPolarity;
}

kInlineFx(kStatus) kHxLight_EnablePowerSaver(kHxLight light, kBool enabled)
{
    kObj(kHxLight, light); 

    obj->powerSaverEnabled = enabled; 

    return kOK; 
}

kInlineFx(kBool) kHxLight_PowerSaverEnabled(kHxLight light)
{
    kObj(kHxLight, light);

    return obj->powerSaverEnabled;
}

kFsFx(k32u) kHxLight_PrimeCount(kHxLight light);

kFsFx(kStatus) kHxLight_PrimePeriod(kHxLight light, k64u* delay, k64u* exposure, k64u* gap);

kInlineFx(kStatus) kHxLight_SetStateCount(kHxLight light, kSize count)
{
    return xkHxLight_VTable(light)->VSetStateCount(light, count);
}

kInlineFx(kSize) kHxLight_StateCount(kHxLight light)
{
    kObj(kHxLight, light); 

    return kArrayList_Count(obj->states);
}

kInlineFx(kHxLightState) kHxLight_StateAt(kHxLight light, kSize index)
{
    kObj(kHxLight, light); 

    return kArrayList_AsT(obj->states, index, kHxLightState);
}

kInlineFx(kStatus) kHxLight_ClearExtensions(kHxLight light)
{
    kObj(kHxLight, light); 

    return kArrayList_Purge(obj->extensions);
}

 //ownership of 'extension' configuration argument not transferred; 
kInlineFx(kStatus) kHxLight_AddExtension(kHxLight light, kLightExt extension)
{
    return xkHxLight_VTable(light)->VAddExtension(light, extension);
}

kInlineFx(kSize) kHxLight_ExtensionCount(kHxLight light)
{
    kObj(kHxLight, light); 

    return kArrayList_Count(obj->extensions);
}

//produces object that must be destroyed by caller
kFsFx(kStatus) kHxLight_ExtensionAt(kHxLight light, kSize index, kLightExt* extension, kAlloc allocator);

kFsFx(kBool) kHxLight_HasOverrideExt(kHxLight light);

kInlineFx(kStatus) kHxLight_Override(kHxLight light, const kLightOverrideItem* overrides, kSize count)
{
    return xkHxLight_VTable(light)->VOverride(light, overrides, count);
}

kInlineFx(kStatus) kHxLight_SetOutputState(kHxLight light, kBool state)
{
    return xkHxLight_VTable(light)->VSetOutputState(light, state);
}

kInlineFx(kStatus) kHxLight_Stats(kHxLight light, kLightStats* stats)
{
    return xkHxLight_VTable(light)->VStats(light, stats);
}

kInlineFx(kStatus) kHxLight_Trigger(kHxLight light)
{
    return xkHxLight_VTable(light)->VTrigger(light);
}

kInlineFx(kStatus) kHxLight_LockOut(kHxLight light, kBool shouldLock)
{
    return xkHxLight_VTable(light)->VLockOut(light, shouldLock);
}

kInlineFx(kStatus) kHxLight_Clear(kHxLight light)
{
    return xkHxLight_VTable(light)->VClear(light);
}

#endif
