/** 
 * @file    kHxCamera.h
 * @brief   Declares the kHxCamera class.
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_FIRESYNC_HX_CAMERA_H
#define K_FIRESYNC_HX_CAMERA_H

#include <kFireSync/kNodeDef.h>
#include <kFireSync/Client/Camera/kCameraInfo.h>

typedef k32s kHxCameraStreamType;

#define kHX_CAMERA_STREAM_TYPE_NULL         (0)
#define kHX_CAMERA_STREAM_TYPE_VIDEO        (1)
#define kHX_CAMERA_STREAM_TYPE_ALGORITHM    (2)

typedef struct kHxCameraDataArgs
{
    kObject data; 
    kObject camera; 
    k32u stateId; 
    kCameraPortMode portMode;
    kHxCameraStreamType type;
    kCameraFrameAlgType algType;
} kHxCameraDataArgs; 

#include <kFireSync/Hardware/kHxCamera.x.h>

/**
 * @class       kHxCamera
 * @extends     kObject
 * @ingroup     kFireSync-Hardware
 * @internal
 * @brief       Abstract base class for hardware camera classes.
 */
//typedef kObject kHxCamera;      --forward-declared in kFsDef.x.h

kInlineFx(kHxNode) kHxCamera_Node(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->node;
}

kInlineFx(kSize) kHxCamera_Index(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->index;
}

kInlineFx(kCameraCapability) kHxCamera_Capabilities(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->capabilities;
}

kInlineFx(kBool) kHxCamera_TrailingRequired(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->isTrailingRequired;
}

kInlineFx(kSize) kHxCamera_FrameSizeGranularity(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->frameSizeGranularity;
}

kInlineFx(kCameraAlgorithm) kHxCamera_Algorithm(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->algorithm;
}

kInlineFx(kCameraInfo) kHxCamera_Info(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->info;
}

kInlineFx(kCameraModel) kHxCamera_Model(kHxCamera camera)
{
    kObj(kHxCamera, camera); 
    
    return kCameraInfo_Model(obj->info); 
}

kInlineFx(k64f) kHxCamera_PixelClockFrequency(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->pixelClockFrequency;
}

kInlineFx(k32u) kHxCamera_ChannelCount(kHxCamera camera)
{
    kObj(kHxCamera, camera);

    return obj->channelCount;
}

kInlineFx(k64f) kHxCamera_PixelPeriod(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return (obj->pixelClockFrequency == 0) ? 0 : (1000000000.0/obj->pixelClockFrequency);
}

kInlineFx(k32u) kHxCamera_PlPipeCount(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->plPipeCount;
}

kInlineFx(kStatus) kHxCamera_SetDataHandler(kHxCamera camera, kCallbackFx function, kPointer receiver)
{
    kObj(kHxCamera, camera); 
    
    obj->dataCallback.function = function; 
    obj->dataCallback.receiver = receiver; 

    return kOK; 
}

kInlineFx(kStatus) kHxCamera_SetVideoBufferSize(kHxCamera camera, kSize requestedSize)
{
    return xkHxCamera_VTable(camera)->VSetVideoBufferSize(camera, requestedSize);
}

kInlineFx(kSize) kHxCamera_VideoBufferSize(kHxCamera camera)
{
    return xkHxCamera_VTable(camera)->VVideoBufferSize(camera);
}

kInlineFx(kStatus) kHxCamera_SetAlgorithmBufferSize(kHxCamera camera, kSize requestedSize)
{
    return xkHxCamera_VTable(camera)->VSetAlgorithmBufferSize(camera, requestedSize);
}

kInlineFx(kSize) kHxCamera_AlgorithmBufferSize(kHxCamera camera)
{
    return xkHxCamera_VTable(camera)->VAlgorithmBufferSize(camera);
}

kInlineFx(kStatus) kHxCamera_AllocateBuffer(kHxCamera camera)
{
    return xkHxCamera_VTable(camera)->VAllocateBuffer(camera);
}

kInlineFx(kBool) kHxCamera_IsConnected(kHxCamera camera)
{
    return xkHxCamera_VTable(camera)->VIsConnected(camera);
}

kInlineFx(kStatus) kHxCamera_Enable(kHxCamera camera, kBool enabled)
{
    kObj(kHxCamera, camera); 

    obj->enabled = enabled; 

    return kOK; 
}

kInlineFx(kBool) kHxCamera_IsEnabled(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->enabled; 
}

kInlineFx(kStatus) kHxCamera_SetControl(kHxCamera camera, kCameraControl type)
{
    kObj(kHxCamera, camera); 

    obj->controlType = type; 

    return kOK; 
}

kInlineFx(kCameraControl) kHxCamera_Control(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->controlType; 
}

kInlineFx(kStatus) kHxCamera_SetControlId(kHxCamera camera, kSize id)
{
    kObj(kHxCamera, camera); 

    obj->controlId = id; 

    return kOK; 
}

kInlineFx(kSize) kHxCamera_ControlId(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->controlId; 
}

kInlineFx(kCameraOutputMode) kHxCamera_OutputCapabilities(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->outputCapabilities;
}

kInlineFx(kStatus) kHxCamera_SetOutputMode(kHxCamera camera, kCameraOutputMode outputMode)
{
    kObj(kHxCamera, camera); 

    obj->outputMode = outputMode; 

    return kOK; 
}

kInlineFx(kCameraOutputMode) kHxCamera_OutputMode(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->outputMode;
}

kInlineFx(kStatus) kHxCamera_SetPortMode(kHxCamera camera, kCameraPortMode portMode)
{
    kObj(kHxCamera, camera); 

    obj->portMode = portMode; 

    return kOK; 
}

kInlineFx(kCameraPortMode) kHxCamera_PortMode(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->portMode;
}

kInlineFx(kStatus) kHxCamera_SetReadout(kHxCamera camera, kCameraReadout readoutMode)
{
    kObj(kHxCamera, camera); 

    obj->readoutMode = readoutMode; 

    return kOK; 
}

kInlineFx(kCameraReadout) kHxCamera_Readout(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->readoutMode;
}

kInlineFx(kStatus) kHxCamera_SetInputCounterSource(kHxCamera camera, kCameraInputCounterSource source)
{
    kObj(kHxCamera, camera); 

    obj->inputCounterSource = source; 

    return kOK; 
}

kInlineFx(kCameraInputCounterSource) kHxCamera_InputCounterSource(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->inputCounterSource; 
}

kInlineFx(kStatus) kHxCamera_SetInputCounterSourceId(kHxCamera camera, k32u id)
{
    kObj(kHxCamera, camera); 

    obj->inputCounterSourceId = id; 

    return kOK; 
}

kInlineFx(k32u) kHxCamera_InputCounterSourceId(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->inputCounterSourceId; 
}

kInlineFx(kStatus) kHxCamera_SetReductionFactor(kHxCamera camera, k32u factor)
{
    kObj(kHxCamera, camera); 

    obj->reductionFactor = factor;

    return kOK; 
}

kInlineFx(k32u) kHxCamera_ReductionFactor(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->reductionFactor;
}

kInlineFx(kStatus) kHxCamera_SetReductionWindow(kHxCamera camera, k32u window)
{
    kObj(kHxCamera, camera); 

    obj->reductionWindow = window;

    return kOK; 
}

kInlineFx(k32u) kHxCamera_ReductionWindow(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->reductionWindow;
}

kInlineFx(kStatus) kHxCamera_SetDuration(kHxCamera camera, k64u duration)
{
    kObj(kHxCamera, camera); 

    obj->duration = duration;

    return kOK; 
}

kInlineFx(k64u) kHxCamera_Duration(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->duration;
}

//returns a period suitable for use in camera priming 
kInlineFx(kStatus) kHxCamera_PrimePeriod(kHxCamera camera, k64u* delay, k64u* exposure, k64u* gap)
{
    kObj(kHxCamera, camera); 

    return kCameraInfo_PrimePeriod(obj->info, delay, exposure, gap);
}

//returns a period suitable for use in camera priming 
kInlineFx(k64u) kHxCamera_PrimePeriod(kHxCamera camera)
{
    k64u delay, exposure, gap;

    kHxCamera_PrimePeriod(camera, &delay, &exposure, &gap);

    return delay + exposure + gap;
}

kInlineFx(k32u) kHxCamera_PrimeCount(kHxCamera camera)
{
    return xkHxCamera_VTable(camera)->VPrimeCount(camera);
}

kInlineFx(kStatus) kHxCamera_EnableRamImage(kHxCamera camera, kBool enabled)
{
    kObj(kHxCamera, camera); 

    obj->ramImageEnabled = enabled; 

    return kOK; 
}

kInlineFx(kBool) kHxCamera_RamImageEnabled(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->ramImageEnabled; 
}

kInlineFx(kStatus) kHxCamera_EnablePrnu(kHxCamera camera, kBool enabled)
{
    kObj(kHxCamera, camera); 

    obj->prnuEnabled = enabled; 

    return kOK; 
}

kInlineFx(kBool) kHxCamera_PrnuEnabled(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->prnuEnabled;
}

kInlineFx(kStatus) kHxCamera_WritePrnu(kHxCamera camera, kImage white, kImage black)
{
    return xkHxCamera_VTable(camera)->VWritePrnu(camera, white, black);
}

kInlineFx(kStatus) kHxCamera_EnableFpn(kHxCamera camera, kBool enabled)
{
    kObj(kHxCamera, camera); 

    obj->fpnEnabled = enabled; 

    return kOK; 
}

kInlineFx(kBool) kHxCamera_FpnEnabled(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->fpnEnabled; 
}

kInlineFx(kStatus) kHxCamera_WriteFpn(kHxCamera camera, kImage offsets)
{
    return xkHxCamera_VTable(camera)->VWriteFpn(camera, offsets);
}

kInlineFx(kStatus) kHxCamera_EnableCalibration(kHxCamera camera, kBool enabled)
{
    kObj(kHxCamera, camera); 

    obj->calibrationEnabled = enabled; 

    return kOK; 
}

kInlineFx(kBool) kHxCamera_CalibrationEnabled(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->calibrationEnabled; 
}

kInlineFx(kStatus) kHxCamera_EnableTestPattern(kHxCamera camera, kBool enabled)
{
    kObj(kHxCamera, camera); 

    obj->testPatternEnabled = enabled; 

    return kOK; 
}

kInlineFx(kBool) kHxCamera_TestPatternEnabled(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->testPatternEnabled; 
}

kInlineFx(kStatus) kHxCamera_CalibrateFpn(kHxCamera camera, kImage image)
{
    return xkHxCamera_VTable(camera)->VCalibrateFpn(camera, image);
}

kInlineFx(kStatus) kHxCamera_WriteRangeLut(kHxCamera camera, const kRangeLutParams* params, const kRangeLutArray* xArray,
                                                             const kRangeLutArray* zArray, const kRangeLutArray* validArray)
{
    return xkHxCamera_VTable(camera)->VWriteRangeLut(camera, params, xArray, zArray, validArray);
}

kInlineFx(kStatus) kHxCamera_WritePhaseDecoderLut(kHxCamera camera, kCameraPhaseDecoderLutType type, kArray1 lut)
{
    return xkHxCamera_VTable(camera)->VWritePhaseDecoderLut(camera, type, lut);
}


kInlineFx(kStatus) kHxCamera_SetAccelerationTestMode(kHxCamera camera, kCameraAccelerationTestMode mode)
{
    kObj(kHxCamera, camera); 

    obj->accelerationTestMode = mode; 

    return kOK; 
}

kInlineFx(kCameraAccelerationTestMode) kHxCamera_AccelerationTestMode(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->accelerationTestMode; 
}

kInlineFx(kStatus) kHxCamera_WriteAccelerationTestData(kHxCamera camera, kObject data)
{
    return xkHxCamera_VTable(camera)->VWriteAccelerationTestData(camera, data);
}

kInlineFx(kStatus) kHxCamera_ReadAccelerationTestResult(kHxCamera camera, kObject* data, kAlloc allocator)
{
    return xkHxCamera_VTable(camera)->VReadAccelerationTestResult(camera, data, allocator);
}

kInlineFx(kStatus) kHxCamera_SetStateCount(kHxCamera camera, kSize count)
{
    return xkHxCamera_VTable(camera)->VSetStateCount(camera, count);
}

kInlineFx(kSize) kHxCamera_StateCount(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return kArrayList_Count(obj->states);
}

kInlineFx(kHxCameraState) kHxCamera_StateAt(kHxCamera camera, kSize index)
{
    kObj(kHxCamera, camera); 

    return kArrayList_AsT(obj->states, index, kHxCameraState);
}

kInlineFx(kStatus) kHxCamera_ClearExtensions(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return kArrayList_Purge(obj->extensions);
}

 //ownership of 'extension' configuration argument not transferred
kInlineFx(kStatus) kHxCamera_AddExtension(kHxCamera camera, kCameraExt extension)
{
    return xkHxCamera_VTable(camera)->VAddExtension(camera, extension);
}

kInlineFx(kSize) kHxCamera_ExtensionCount(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return kArrayList_Count(obj->extensions);
}

//produces object that must be destroyed by caller
kFsFx(kStatus) kHxCamera_ExtensionAt(kHxCamera camera, kSize index, kCameraExt* extension, kAlloc allocator);

kFsFx(kBool) kHxCamera_IsExtensionEnabled(kHxCamera camera, kCameraExtension id);

kInlineFx(kStatus) kHxCamera_Override(kHxCamera camera, const kCameraOverrideItem* overrides, kSize count)
{
    return xkHxCamera_VTable(camera)->VOverride(camera, overrides, count);
}

//callee should add ref to retain image, if desired
kInlineFx(kStatus) kHxCamera_SendDynamicFpnImage(kHxCamera camera, kImage image)
{
    return xkHxCamera_VTable(camera)->VSendDynamicFpnImage(camera, image);
}

kInlineFx(kSize) kHxCamera_ImagerRegisterCapacity(kHxCamera camera)
{
    return xkHxCamera_VTable(camera)->VImagerRegisterCapacity(camera);
}

kInlineFx(k32u) kHxCamera_ReadImagerRegister(kHxCamera camera, kSize registerIndex)
{
    return xkHxCamera_VTable(camera)->VReadImagerRegister(camera, registerIndex);
}

kInlineFx(kStatus) kHxCamera_WriteImagerRegister(kHxCamera camera, kSize registerIndex, k32u value)
{
    return xkHxCamera_VTable(camera)->VWriteImagerRegister(camera, registerIndex, value);
}

kInlineFx(k32u) kHxCamera_LvdsPower(kHxCamera camera)
{
    kObj(kHxCamera, camera); 

    return obj->lvdsPower;
}

kInlineFx(kStatus) kHxCamera_SetLvdsPower(kHxCamera camera, k32u power, kBool commit)
{
    kObj(kHxCamera, camera); 

    obj->lvdsPower = power;

    return xkHxCamera_VTable(camera)->VSetLvdsPower(camera, power, commit);
}

kInlineFx(kStatus) kHxCamera_Calibrate(kHxCamera camera)
{
    return xkHxCamera_VTable(camera)->VCalibrate(camera);
}

kInlineFx(kStatus) kHxCamera_Stats(kHxCamera camera, kCameraStats* stats)
{
    return xkHxCamera_VTable(camera)->VStats(camera, stats);
}

kInlineFx(kStatus) kHxCamera_Trigger(kHxCamera camera)
{
    return xkHxCamera_VTable(camera)->VTrigger(camera);
}

kInlineFx(kStatus) kHxCamera_Clear(kHxCamera camera)
{
    return xkHxCamera_VTable(camera)->VClear(camera);
}

#endif
