/** 
 * @file    kHxAnalogOut.h
 * @brief   Declares the kHxAnalogOut class.
 *
 * @internal
 * Copyright (C) 2018-2022 by LMI Technologies Inc. All rights reserved.
 */
#ifndef K_FIRESYNC_HX_ANALOG_OUT_H
#define K_FIRESYNC_HX_ANALOG_OUT_H

#include <kFireSync/kNodeDef.h>
#include <kFireSync/Hardware/kHxAnalogOut.x.h>

/**
 * @class       kHxAnalogOut
 * @extends     kObject
 * @ingroup     kFireSync-Hardware
 * @internal
 * @brief       Abstract base class for hardware analog output classes.
 */
//typedef kObject kHxAnalogOut;      --forward-declared in kFsDef.x.h

kInlineFx(kHxNode) kHxAnalogOut_Node(kHxAnalogOut analog)
{
    kObj(kHxAnalogOut, analog); 

    return obj->node;
}

kInlineFx(kSize) kHxAnalogOut_Index(kHxAnalogOut analog)
{
    kObj(kHxAnalogOut, analog); 

    return obj->index;
}

kInlineFx(kAnalogOutModel) kHxAnalogOut_AnalogOutModel(kHxAnalogOut analog)
{
    kObj(kHxAnalogOut, analog); 

    return obj->model;
}

kInlineFx(kStatus) kHxAnalogOut_Enable(kHxAnalogOut analog, kBool enabled)
{
    kObj(kHxAnalogOut, analog); 

    obj->enabled = enabled; 

    return kOK; 
}

kInlineFx(kBool) kHxAnalogOut_IsEnabled(kHxAnalogOut analog)
{
    kObj(kHxAnalogOut, analog); 

    return obj->enabled; 
}

kInlineFx(kStatus) kHxAnalogOut_SetControl(kHxAnalogOut analog, kAnalogOutControl type)
{
    kObj(kHxAnalogOut, analog); 

    obj->controlType = type; 

    return kOK; 
}

kInlineFx(kAnalogOutControl) kHxAnalogOut_Control(kHxAnalogOut analog)
{
    kObj(kHxAnalogOut, analog); 

    return obj->controlType; 
}

kInlineFx(kStatus) kHxAnalogOut_SetControlId(kHxAnalogOut analog, kSize id)
{
    kObj(kHxAnalogOut, analog); 

    obj->controlId = id; 

    return kOK; 
}

kInlineFx(kSize) kHxAnalogOut_ControlId(kHxAnalogOut analog)
{
    kObj(kHxAnalogOut, analog); 

    return obj->controlId; 
}

/**
 * Sets (or clears) calibration information.
 *
 * The calibration table provides a mapping between nominal current and calibrated current. 
 * The table should have the following properties: 
 * * kArray2<k64f>  
 * * First column represents nominal current.
 * * Second column represents calibrated current.
 * * At least two rows with unique nominal/calibrated current values.
 *
 * Calibration is optional and can be cleared by calling this function with a null calibration object.
 * If calibration is not provided, output will governed by the nominal device calibration model
 * (e.g., 0 - 22 mA, mapped directly to DAC codes).
 *
 * Calibration is transient (will not be saved to storage). Reapply calibration as needed
 * after reset. 
 *
 * @public              @memberof kHxAnalogOut
 * @param   analog      Analog output object.
 * @param   calibration Calibration table (kArray2<k64f>), or null to clear calibration.
 * @return              Minimum current (mA).
 */
kInlineFx(kStatus) kHxAnalogOut_SetCalibration(kHxAnalogOut analog, kArray2 calibration)
{
    return xkHxAnalogOut_VTable(analog)->VSetCalibration(analog, calibration);
}

/**
 * Gets the minimum supported current value. 
 *
 * Before calibration the minimum current is determined by nominal device capability. 
 * After calibration, the minimum current is determined by using the provided calibration table
 * to estimate the calibrated current corresponding the minimum DAC code.  
 *
 * @public              @memberof kHxAnalogOut
 * @param   analog      Analog output object.
 * @return              Minimum current (mA).
 */
kInlineFx(k64f) kHxAnalogOut_MinCurrent(kHxAnalogOut analog)
{
    return xkHxAnalogOut_VTable(analog)->VMinCurrent(analog);
}

/**
 * Gets the maximum supported current value. 
 *
 * Before calibration the maximum current is determined by nominal device capability. 
 * After calibration, the maximum current is determined by using the provided calibration table
 * to estimate the calibrated current corresponding the maximum DAC code.  
 *
 * @public              @memberof kHxAnalogOut
 * @param   analog      Analog output object.
 * @return              Maximum current (mA).
 */
kInlineFx(k64f) kHxAnalogOut_MaxCurrent(kHxAnalogOut analog)
{
    return xkHxAnalogOut_VTable(analog)->VMaxCurrent(analog);
}

/**
* Reports whether this device is currently calibrated. 
*
* @public              @memberof kHxAnalogOut
* @param   analog      Analog output object.
* @return              kTRUE if calibrated. 
*/
kInlineFx(kBool) kHxAnalogOut_IsCalibrated(kHxAnalogOut analog)
{
    return xkHxAnalogOut_VTable(analog)->VIsCalibrated(analog);
}

/**
* Enqueues a current value, in milliamps, for output.
*
* Before calibration, the current is interpreted according to nominal device capability.
* After calibration, the current is interpreted according to the provided calibration table.
*
* If the specified current is outside the range of the device, the emitted current will be
* clamped to device limits.
*
* @public              @memberof kHxAnalogOut
* @param   analog      Analog output object.
* @param   current     Current value to be enqueued, in milliamps.
* @return              Operation status.
*/
kInlineFx(kStatus) kHxAnalogOut_EnqueueCurrent(kHxAnalogOut analog, k64f current)
{
    return xkHxAnalogOut_VTable(analog)->VEnqueueCurrent(analog, current);
}

kInlineFx(kStatus) kHxAnalogOut_Stats(kHxAnalogOut analog, kAnalogOutStats* stats)
{
    return xkHxAnalogOut_VTable(analog)->VStats(analog, stats);
}

kInlineFx(kStatus) kHxAnalogOut_Clear(kHxAnalogOut analog)
{
    return xkHxAnalogOut_VTable(analog)->VClear(analog);
}

#endif
