/** 
 * @file    kIpp.h
 * @brief   Declares the kIpp type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_IPP_H_INCLUDED
#define K_FIRESYNC_IPP_H_INCLUDED

#include <kFireSync/kFsDef.h>

/**
* @class   kIpp
* @extends kObject
* @ingroup kFireSync-Data
* @brief   Collection of image processing utility functions.
*/
typedef kObject kIpp;

/** 
 * Demosaics a greyscale, bayer-masked image to generate a color image. 
 *
 * This method uses the kImage_Cfa property to determine the bayer mask.
 * 
 * The output image size can vary, depending on the demosaic style.
 * 
 * @public              @memberof kIpp
 * @param   input       Input image (kImage<k8u>).
 * @param   output      Output image (kImage<kRgb>) pre-allocated with correct dimensions.
 * @param   style       Demosaic style.
 * @return              Operation status.
 */
kFsFx(kStatus) kIpp_Demosaic(kImage input, kImage output, kDemosaicStyle style); 

/** 
 * Resamples an input image to generate a scaled, shifted output image.
 * 
 * @public              @memberof kIpp
 * @param   input       Input image (kImage<k8u>).
 * @param   output      Output image (kImage<k8u>) pre-allocated with correct dimensions.
 * @param   temp        Temporary data structure required by algorithm. 
 * @param   xs          X scale factor.
 * @param   ys          Y scale factor.
 * @param   xOrigin     X origin.
 * @param   yOrigin     Y origin.
 * @return              Operation status.
 */
kFsFx(kStatus) kIpp_ResizeShift(kImage input, kImage output, kArray1 temp, k64f xs, k64f ys, k64f xOrigin, k64f yOrigin);

/** 
 * Mirrors an input image in the X and/or Y axes.
 * 
 * @public              @memberof kIpp
 * @param   input       Input image (kImage<k8u>).
 * @param   output      Output image (kImage<k8u>) pre-allocated with correct dimensions, or kNULL for in-place.  
 * @param   axes        Axes to mirror. 
 * @return              Operation status.
 */
kFsFx(kStatus) kIpp_Mirror(kImage input, kImage output, kAxis axes);

/** 
 * Transposes an image.
 * 
 * @public              @memberof kIpp
 * @param   input       Input image (kImage).
 * @param   output      Output image (kImage) pre-allocated with correct dimensions (width is input height and height is input width). Self-transposing not supported.
 * @return              Operation status.
 */
kFsFx(kStatus) kIpp_Transpose(kImage input, kImage output);

/** 
 * Bit-packs an input image. 
 * 
 * Input pixels greater than 0 are represented by '1'. 
 * 
 * The output image should be allocated with the correct size before calling this
 * function.
 * 
 * @public              @memberof kIpp
 * @param   input       Input image (kImage<k8u>).
 * @param   output      Output image (kImage<k8u>) pre-allocated with correct dimensions.
 * @return              Operation status.
 */
kFsFx(kStatus) kIpp_Pack(kImage input, kImage output);

/** 
 * Bit-unpacks an input image. 
 * 
 * Input bits are mapped to 0 or 255. 
 * 
 * The output image should be allocated with the correct size before calling this
 * function.
 * 
 * @public              @memberof kIpp
 * @param   input       Input image (kImage<k8u>).
 * @param   output      Output image (kImage<k8u>) pre-allocated with correct dimensions.
 * @return              Operation status.
 */
kFsFx(kStatus) kIpp_Unpack(kImage input, kImage output);

#include <kFireSync/Data/kIpp.x.h>

#endif
