/** 
 * @file    kGraphic.x.h
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_GRAPHIC_X_H
#define K_FIRESYNC_GRAPHIC_X_H

#include <kFireSync/Data/kGraphic.h>

#define kGRAPHIC_MIN_NODE_BYTES         (4096)
#define kGRAPHIC_DEFAULT_STRING_BYTES   (256)

kInlineFx(kSize) xkGraphic_Align(kSize offset)
{
    return kSize_Align(offset, kALIGN_ANY);
}

typedef struct kGraphicNode 
{
    kPointer next; 
    k32u byteCapacity;
    k32u byteCount;
} kGraphicNode; 

#define kGRAPHIC_NODE_HEADER_SIZE               (xkGraphic_Align(sizeof(kGraphicNode)))

kInlineFx(void*) xkGraphic_NodeDataAt(kGraphicNode* node, kSize index)
{
    return kPointer_ByteOffset(node, kGRAPHIC_NODE_HEADER_SIZE + index); 
}

kInlineFx(void*) xkGraphic_NodeDataBegin(kGraphicNode* node)
{
    return xkGraphic_NodeDataAt(node, 0); 
}

kInlineFx(void*) xkGraphic_NodeDataEnd(kGraphicNode* node)
{
    return xkGraphic_NodeDataAt(node, node->byteCount); 
}

typedef struct kGraphicClass 
{
    kObjectClass base; 
    kPointer head; 
    kPointer tail; 
    k32u nextResId; 
    k32u byteCapacity; 
} kGraphicClass; 

kDeclareClassEx(kFs, kGraphic, kObject)

kFsFx(kStatus) kGraphic_Init(kGraphic graphic, kType classType, kAlloc allocator);
kFsFx(kStatus) kGraphic_VClone(kGraphic graphic, kGraphic source, kAlloc valueAlloc, kObject context); 
kFsFx(kStatus) kGraphic_Release(kGraphic graphic); 

kFsFx(kStatus) xkGraphic_WriteDat5V1(kGraphic graphic, kSerializer serializer);
kFsFx(kStatus) xkGraphic_ReadDat5V1(kGraphic graphic, kSerializer serializer);
kFsFx(kStatus) xkGraphic_WriteDat6V0(kGraphic graphic, kSerializer serializer);
kFsFx(kStatus) xkGraphic_ReadDat6V0(kGraphic graphic, kSerializer serializer);
kFsFx(kStatus) kGraphic_VRelease(kGraphic graphic);
kFsFx(kSize) kGraphic_VSize(kGraphic graphic);

/** 
 * Draws a set of rotated ellipses.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   rects       Array of bounding rectangles (if null, will allocate a buffer without copying rectangles).    
 * @param   count       Count of bounding rectangles.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kRotatedRect32s*) kGraphic_DrawRotatedEllipses32s(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, const kRotatedRect32s* rects, kSize count);

/** 
 * Draws a set of rotated ellipses.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   brush       Brush to create fill.   
 * @param   rects       Array of bounding rectangles (if null, will allocate a buffer without copying rectangles).    
 * @param   count       Count of bounding rectangles.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kRotatedRect32f*) kGraphic_DrawRotatedEllipses32f(kGraphic graphic, kGraphicPen pen, kGraphicBrush brush, const kRotatedRect32f* rects, kSize count);

/** 
 * Draws connected curve segments.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   points      Array of curve points (if null, will allocate a buffer without copying points).    
 * @param   count       Count of points.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kPoint32s*) kGraphic_DrawCurve32s(kGraphic graphic, kGraphicPen pen, const kPoint32s* points, kSize count);

/** 
 * Draws connected curve segments.
 *
 * The internal buffer returned by this function can be modifed. 
 *
 * @public              @memberof kGraphic
 * @param   graphic     Graphic object.   
 * @param   pen         Pen to create strokes.   
 * @param   points      Array of curve points (if null, will allocate a buffer without copying points).    
 * @param   count       Count of points.    
 * @return              Returns pointer to buffer holding the new item(s).
 */
kFsFx(kPoint32f*) kGraphic_DrawCurve32f(kGraphic graphic, kGraphicPen pen, const kPoint32f* points, kSize count);

typedef kObject kGraphicIt;

typedef struct kGraphicItClass
{
    kObjectClass base; 
    kGraphicNode* node; 
    k32u offset; 
} kGraphicItClass;

/** 
 * Constructs kGraphicIt object
 *
 * @public              @memberof kGraphic 
 * @param   graphic     Graphic object.   
 * @param   it          Receives object
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kGraphic_GetIterator(kGraphic graphic, kGraphicIt* it, kObject allocator);

kFsFx(kIterator) kGraphicIt_GetIterator(kGraphicIt it);
kFsFx(kBool) kGraphicIt_HasNext(kGraphicIt it, kIterator iterator);
kFsFx(void*) kGraphicIt_Next(kGraphicIt it, kIterator* iterator);

kDeclareClassEx(kFs, kGraphicIt, kObject)

typedef struct kGraphicEntry  
{
    k32u byteCount; 
} kGraphicEntry; 

#define kGRAPHIC_ENTRY_HEADER_SIZE          (xkGraphic_Align(sizeof(kGraphicEntry)))

typedef enum
{
    kGRAPHIC_TYPE_UNKNOWN,
    kGRAPHIC_TYPE_SOLID_BRUSH,
    kGRAPHIC_TYPE_PEN,
    kGRAPHIC_TYPE_FONT,
    kGRAPHIC_TYPE_MARKER, 
    kGRAPHIC_TYPE_POLYLINES_32S,
    kGRAPHIC_TYPE_POLYLINES_32F,
    kGRAPHIC_TYPE_CURVES_32S,
    kGRAPHIC_TYPE_CURVES_32F,
    kGRAPHIC_TYPE_POLYGONS_32S,
    kGRAPHIC_TYPE_POLYGONS_32F,
    kGRAPHIC_TYPE_RECTS_32S, 
    kGRAPHIC_TYPE_RECTS_32F, 
    kGRAPHIC_TYPE_ELLIPSES_32S,
    kGRAPHIC_TYPE_ELLIPSES_32F,
    kGRAPHIC_TYPE_ROTATED_RECTS_32S,
    kGRAPHIC_TYPE_ROTATED_RECTS_32F,
    kGRAPHIC_TYPE_ROTATED_ELLIPSES_32S,
    kGRAPHIC_TYPE_ROTATED_ELLIPSES_32F,
    kGRAPHIC_TYPE_POINTS_32S,
    kGRAPHIC_TYPE_POINTS_32F,
    kGRAPHIC_TYPE_STRING_32S,
    kGRAPHIC_TYPE_STRING_32F,
    kGRAPHIC_TYPE_IMAGE_32S,
    kGRAPHIC_TYPE_IMAGE_32F
} kGraphicType; 

#define kGRAPHIC_ITEM_FIELDS()                      \
    kGraphicType    type

typedef struct kGraphicItem 
{
    kGRAPHIC_ITEM_FIELDS(); 
} kGraphicItem; 

#define kGRAPHIC_RESOURCE_FIELDS()                  \
    kGRAPHIC_ITEM_FIELDS();                         \
    k32u id

typedef struct kGraphicResource 
{
    kGRAPHIC_RESOURCE_FIELDS(); 
} kGraphicResource; 

kInlineFx(k32u) xkGraphicResource_Id(kPointer resource)
{
    const kGraphicResource* r = kCast(kGraphicResource*, resource); 

    return kIsNull(r) ? 0 : r->id; 
}

typedef struct kGraphicSolidBrushClass 
{
    kGRAPHIC_RESOURCE_FIELDS(); 
    kColor color; 
} kGraphicSolidBrushClass; 

typedef struct kGraphicPenClass 
{
    kGRAPHIC_RESOURCE_FIELDS(); 
    k32u brush; 
    k32f width; 
} kGraphicPenClass; 

typedef struct kGraphicFontClass 
{
    kGRAPHIC_RESOURCE_FIELDS(); 
    kFontFamily family;
    kFontStyle style;
    k32f height;
} kGraphicFontClass; 

typedef struct kGraphicMarkerClass 
{
    kGRAPHIC_RESOURCE_FIELDS(); 
    kMarkerShape shape;
    k32f size;
} kGraphicMarkerClass; 

typedef struct kGraphicShapes 
{
    kGRAPHIC_ITEM_FIELDS(); 
    k32u pen; 
    k32u brush;
    k32u shapeCount;
    k32u itemsPerShape; 
} kGraphicShapes; 

kFsFx(void*) kGraphicShapes_Data(kGraphicShapes* item, kType* type); 

typedef struct kGraphicPoints 
{
    kGRAPHIC_ITEM_FIELDS(); 
    k32u pen; 
    k32u brush;
    k32u marker; 
    k32u count; 
} kGraphicPoints; 

kFsFx(void*) kGraphicPoints_Data(kGraphicPoints* item, kType* type); 

typedef struct kGraphicString32s 
{
    kGRAPHIC_ITEM_FIELDS(); 
    k32u font; 
    k32u brush; 
    kPoint32s location;
    k32s angle; 
    kAlignment alignment;
    k32u byteCount; 
} kGraphicString32s; 

kFsFx(kChar*) kGraphicString32s_Data(kGraphicString32s* item); 

typedef struct kGraphicString32f 
{
    kGRAPHIC_ITEM_FIELDS(); 
    k32u font; 
    k32u brush; 
    kPoint32f location;
    k32f angle; 
    kAlignment alignment;
    k32u byteCount; 
} kGraphicString32f; 

kFsFx(kChar*) kGraphicString32f_Data(kGraphicString32f* item); 

typedef struct kGraphicImage32s 
{
    kGRAPHIC_ITEM_FIELDS(); 
    kType pixelType; 
    k32u width; 
    k32u height; 
    kRect32s dest;  
} kGraphicImage32s; 

kFsFx(void*) kGraphicImage32s_Data(kGraphicImage32s* item); 

typedef struct kGraphicImage32f 
{
    kGRAPHIC_ITEM_FIELDS(); 
    kType pixelType; 
    k32u width; 
    k32u height; 
    kRect32f dest;  
} kGraphicImage32f; 

kFsFx(void*) kGraphicImage32f_Data(kGraphicImage32f* item); 

#endif
