/** 
 * @file    kCudaStream.x.h
 *
 * @internal
 * Copyright (C) 2019-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CUDA_STREAM_X_H
#define K_FIRESYNC_CUDA_STREAM_X_H

#include <kApi/kObject.h>

#define kCUDA_STREAM_INVALID        (~0)

typedef struct kCudaStreamVTable
{
    kObjectVTable base;
} kCudaStreamVTable;

typedef struct kCudaStreamStatic
{
    kObject defaultStream;      //default Cuda stream
} kCudaStreamStatic;

typedef struct kCudaStreamClass
{
    kObjectClass base;
    kPointer handle;
    kBool isDefault;
    kCudaSync sync;
} kCudaStreamClass;

kDeclareFullClassEx(kFs, kCudaStream, kObject)

kFsFx(kStatus) xkCudaStream_InitStatic();
kFsFx(kStatus) xkCudaStream_ReleaseStatic();

kFsFx(kStatus) kCudaStream_ConstructDefault(kCudaStream* cudaStream, kObject allocator);

kFsFx(kStatus) xkCudaStream_Init(kCudaStream cudaStream, kType classType, kObject allocator);
kFsFx(kStatus) xkCudaStream_InitDefault(kCudaStream cudaStream, kType classType, kObject allocator);
kFsFx(kStatus) xkCudaStream_VRelease(kCudaStream cudaStream);

/** 
 * Gets the default Cuda stream.
 *
 * @public      @memberof kCudaStream
 * @return      Default Cuda stream. 
 */
kInlineFx(kCudaStream) xkCudaStream_Default()
{
    return xkCudaStream_Static()->defaultStream;
}

/** 
 * Returns the passed Cuda stream, or if null, the default Cuda stream. 
 *
 * @public              @memberof kCudaStream
 * @param   stream      Cuda stream. 
 * @return              Cuda stream. 
 */
kInlineFx(kCudaStream) xkCudaStream_Fallback(kCudaStream stream)
{
    return kIsNull(stream) ? xkCudaStream_Default() : stream; 
}

/* 
* Deprecated (Stage 1): not recommended for further use, but not yet announced via kDeprecate
*/

//[Deprecated] Use of the default stream is deprecated; refactor code to use a dedicated stream. 
#define kCudaStream_Default xkCudaStream_Default

//[Deprecated] Reliance on the default stream is deprecated; refactor code to use a dedicated stream. 
#define kCudaStream_Fallback xkCudaStream_Fallback

#endif
