/** 
 * @file    kCudaDevice.h
 * @brief   Collection of Cuda device management functions.
 *
 * @internal
 * Copyright (C) 2020-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CUDA_DEVICE_H
#define K_FIRESYNC_CUDA_DEVICE_H

#include <kFireSync/Cuda/kCudaDef.h>
#include <kFireSync/Cuda/kCudaDevice.x.h>

/**
 * @class   kCudaDevice
 * @ingroup kFireSync-Cuda
 * @brief   Collection of Cuda device management functions.
 */

/** 
 * Reports the number of Cuda-capable devices present in the system.
 *
 * @public      @memberof kCudaDevice
 * @return      Count of Cuda-capable devices.
 */
kFsFx(kSize) kCudaDevice_Count();

/** 
 * Reports the index of the currently selected device.
 *
 * @public          @memberof kCudaDevice
 * @param   index   Receives index of currently selected device.
 * @return          Operation status.
 */
kFsFx(kStatus) kCudaDevice_Selected(kSize* index);

/** 
 * Reports the properties associated with a Cuda device. 
 *
 * @public              @memberof kCudaDevice
 * @param   index       Cuda device index (kSIZE_NULL for currently selected device).
 * @param   properties  Receives properties.
 * @return              Operation status.
 */
kInlineFx(kStatus) kCudaDevice_Properties(kSize index, kCudaProperties* properties)
{
    return xkCudaDevice_Properties(index, properties);
}

/** 
 * Reports the properties associated with the currently selected Cuda device. 
 *
 * @public              @memberof kCudaDevice
 * @param   properties  Receives properties.
 * @return              Operation status.
 */
kInlineFx(kStatus) kCudaDevice_Properties(kCudaProperties* properties)
{
    return xkCudaDevice_Properties(kSIZE_NULL, properties);
}

/** 
 * Reports the minimum compute capability recommended for use with FireSync platform Cuda support.
 *
 * @public      @memberof kCudaDevice
 * @return      Minimum recommended vesrion.
 */
kInlineFx(kVersion) kCudaDevice_MinimumComputeCapability()
{
    return kVersion_Create(xkCUDA_MIN_COMPUTE_CAPABILITY_MAJOR, xkCUDA_MIN_COMPUTE_CAPABILITY_MINOR, 0, 0);
}

/** 
 * Reports whether a device is selected and, optionally, compatible with default minimimum platform requriements.
 *
 * Note, the kCudaUtils_EnvironmentStatus function can be used to help determine the reason that a Cuda device 
 * is not available.
 * 
 * @public                      @memberof kCudaDevice
 * @param   checkCompatibility  If kTRUE, checks compatibility; otherwise, checks only for presence of any device.
 * @return                      kTRUE if compatible device is available. 
 * @see                         kCudaUtils_EnvironmentStatus
 */
kFsFx(kBool) kCudaDevice_IsAvailable(kBool checkCompatibility = kTRUE);

#endif
