/** 
 * @file    kPipe.x.h
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PIPE_X_H
#define K_FIRESYNC_PIPE_X_H

#define kPIPE_DEFAULT_MAX_QUEUE_SIZE          (1 << 23)      //Default maximum pipe queue size (bytes).
#define kPIPE_DEFAULT_MAX_QUEUE_COUNT         (1 << 12)      //Default maximum pipe queue count (items).
#define kPIPE_DEFAULT_TIMER_PERIOD_US         (1000u)        //Default timer block callback period, in microseconds

typedef struct kPipeClass
{
    kObjectClass base; 

    kNode node;                                 //Node (hierarchy ancestor).
    kSize maxQueueSize;                         //Maximum total message input queue data size (bytes). 
    kSize maxQueueCount;                        //Message message input queue count (items). 
    kPipeEngine engineType;                     //Pipe engine type. 
    
    kSize threadCount;                          //Desired thread count. 
    kThreadPriorityClass threadPriorityClass;   //Desired thread priority class (high, normal, low).
    k32s threadPriorityOffset;                  //Desired thread priority offset within class.
    kBitArray threadAffinity;                   //Desired thread affinity.

    kArrayList blockTypes;                      //User block type information -- kArraylist<kBlockTypeInfo>.    
    kArrayList systemBlocks;                    //Blocks added automatically by system -- kArrayList<kBlock>.
    kArrayList userBlocks;                      //Blocks added by user -- kArrayList<kBlock>.       
    kArrayList routes;                          //Messages routes -- kArrayList<kRoute>. 

    kBlock timerBlock;                          //Timer block object.
    kBool timerEnabled;                         //Is timer block enabled.
    k64u timerPeriod;                           //Timer block period.
} kPipeClass; 

kDeclareClassEx(kFs, kPipe, kObject)
        
kFsFx(kStatus) kPipe_Construct(kPipe* pipe, kNode node, kAlloc allocator); 

kFsFx(kStatus) kPipe_Init(kPipe pipe, kType type, kNode node, kAlloc alloc); 
kFsFx(kStatus) kPipe_VRelease(kPipe pipe); 

kFsFx(kStatus) kPipe_Parse(kPipe pipe, kXml xml, kXmlItem item); 
kFsFx(kStatus) kPipe_ParseTypes(kPipe pipe, kXml xml, kXmlItem item); 
kFsFx(kStatus) kPipe_ParseBlocks(kPipe pipe, kXml xml, kXmlItem item, kArrayList blockList); 
kFsFx(kStatus) kPipe_ParseRoutes(kPipe pipe, kXml xml, kXmlItem item); 

kFsFx(kStatus) kPipe_Format(kPipe pipe, kXml xml, kXmlItem item); 
kFsFx(kStatus) kPipe_FormatBlocks(kPipe pipe, kXml xml, kXmlItem item, const kChar* itemName, kArrayList blockList); 
kFsFx(kStatus) kPipe_FormatRoutes(kPipe pipe, kXml xml, kXmlItem item); 

kFsFx(kStatus) kPipe_VerificationLabel(kPipe pipe, kChar* label, kSize capacity); 
kFsFx(kStatus) kPipe_Verify(kPipe pipe); 

kFsFx(kStatus) kPipe_ClearEx(kPipe pipe, kNodeClearOption options); 

kFsFx(kStatus) kPipe_ClearBlocks(kPipe pipe); 
kFsFx(kStatus) kPipe_ClearAffinity(kPipe pipe);

kFsFx(kStatus) kPipe_AddSystemBlock(kPipe pipe, kBlock block); 
kFsFx(kStatus) kPipe_RemoveSystemBlock(kPipe pipe, kBlock block);

kFsFx(k32u) kPipe_NextBlockId(kPipe pipe); 
kFsFx(kBool) kPipe_IsConcurrent(kPipe pipe);

kFsFx(kStatus) xkPipe_RegisterTimerBlock(kPipe pipe);
kFsFx(kStatus) xkPipe_UnregisterTimerBlock(kPipe pipe);

#endif
