/** 
 * @file    kNode.x.h
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_NODE_X_H
#define K_FIRESYNC_NODE_X_H

#include <kFireSync/Client/Network/kDiscoveryProvider.h>

#define kNODE_DEFAULT_ID                        (10)            //Default device id (historical convention). 
#define kNODE_MIN_ID                            (1)             //Minimum valid node id. 
#define kNODE_MAX_ID                            (0xFFFFFFFF)    //Maximum valid node id. 

#define kNODE_DISCOVERY_CHECK_COUNT             (3)             //Number of failed discovery checks before sensor considered absent. 
#define kNODE_ACTIVITY_CHECK_WINDOW             (3)       
#define kNODE_ADDRESS_CHECK_WINDOW              (3)             //Number of discovery cycles with address change before existing address considered stale.

#define kNODE_CONSISTENCY_INTERVAL              (4000000)       //Hold time, after foreground info is updated, 
                                                                //before consistency with background info is checked (us). 

#define kNODE_CONNECT_TIMEOUT                   (5000000)       //Time to retry connection attempts before giving up. 

#define kNODE_DEVICE_CONFIG_VERSION             (1)             //Device config format version.
#define kNODE_MODE_CONFIG_VERSION               (1)             //Mode config format version.

#define kNODE_VIRTUAL_CONFIG_VERSION            (1)             //Virtual device hardware configuration format version.

#define kNODE_DEFAULT_POWER_SAVER_TIMEOUT      (300000000)      //Default power saver timeout.
#define kNODE_DEFAULT_POWER_SAVER_THRESHOLD    (20)             //Default power saver threshold.

#define kNODE_HISTORY_COUNT                    (8)              //Count of discovery history information to maintain.
    
typedef kStatus (kCall * kNodeConstructModuleFx)(kObject* module, kNode node, kSize index, kAlloc allocator); 
typedef kStatus (kCall * kNodeParseModuleFx)(kObject module, kXml xml, kXmlItem item); 
typedef kStatus (kCall * kNodeFormatModuleFx)(kObject module, kXml xml, kXmlItem item); 

typedef struct kNodeClass
{
    kObjectClass base; 

    kSystem system;                                     //System object (parent).
    kNodeProvider provider;                             //Control/health/data communication interface.    
    kDiscoveryProvider discoveryProvider;               //Discovery communication interface.

    //begin synchronized (system state lock)
    kDiscoveryInfo discoveryInfo;                       //Latest discovery info.
    kBool discoveryHistory[kNODE_HISTORY_COUNT];        //Discovery presence history (^2).
    k64u discoveryCount;                                //Discovery presence history count.
    kBool addressHistory[kNODE_HISTORY_COUNT];          //Address change history history (^2).
    k64u addressHistoryCount;                           //Address change history count count.
    kBool activityHistory[kNODE_HISTORY_COUNT];         //Communication activity history (^2).
    k64u activityHistoryCount;                          //Communication activity history count.
    k32s activityPreviousValue;                         //Value of activity counter at last check. 
    kNodeState runState;                                //Current node state, as reported via control channel.
    k64u stateUpdateTime;                               //Time of last control channel update of state info.
    k64u configRevision;                                //Configuration revision id for currently-loaded configuration.
    k64u configUpdateTime;                              //Time of last update of config revision.
    kBool isResetting;                                  //Is the node currently resetting?
    k64u resetStartTime;                                //Local time at which reset operation was started (us). 
    k64u resetHoldInterval;                             //Duration of reset operation (us).
    kBool isCancelled;                                  //Was i/o cancelled by user?
    kBool isConnected;                                  //Is node currently connected?
    kBool isCompatible;                                 //Is node protocol compatible with client protocol?
    kBool hasCommError;                                 //Has a communication error occurred?
    kNodeInfo nodeInfo;                                 //General node information. 
    //end synchronized (system state lock)

    kStatus beginConnectStatus;                         //Communicates state between BeginConnect/EndConnect.

    k32u nextId;                                        //Id that will take effect after reset (device config).    
    kBootConfig bootConfig;                             //Boot configuration parameters (device config). 
    kSize localDigitalInputCount;                       //Supported count of local (node) digital inputs (used for exposure trigger validation).
    kSize networkDigitalInputCount;                     //Supported count of network (master) digital inputs (used for exposure trigger validation).
    kBool hasExternalTemp;                              //Specifies whether device has external temperature probe.
    kSize fanCount;                                     //Number of fans present.
    kText128 appNameTemp;                               //Required for implementation of kNode_AppName

    kAtomic32s activityCounter;                         //Incremented when health or data messages are received (atomic). 

    kArrayList errors;                                  //Errors detected during previous verification -- kArrayList<kText128>.
    kArrayList warnings;                                //Warnings detected during previous verification -- kArrayList<kText128>.

    kArrayList healthStats;                             //Health statistics retrieved via polling -- kArrayList<kHealthStat>. 

    kArrayList dataPorts;                               //Remote data ports to which client will attach in kNode_OpenData. 
   
    kXml deviceConfigXml;                               //Temp variable used during config read/write. 
    kBool deviceConfigModified;                         //Has device config been locally modified?
    kXml modeConfigXml;                                 //Temp variable used during config read/write. 
    kBool modeConfigModified;                           //Has mode config been locally modified?

    kBool enabled;                                      //Is node enabled?

    kBool hasThreadAffinity;                            //Does the node support setting thread affinity?
    kThreadPriorityClass minThreadClass;                //Minimum thread priority class supported by node.
    kThreadPriorityClass maxThreadClass;                //Maximum thread priority class supported by node.
    k32s minLowThreadOffset;                            //Minimum Low-priority thread offset supported by node.
    k32s maxLowThreadOffset;                            //Maximum Low-priority thread offset supported by node.
    k32s minNormalThreadOffset;                         //Minimum Normal-priority thread offset supported by node.
    k32s maxNormalThreadOffset;                         //Maximum Normal-priority thread offset supported by node.
    k32s minHighThreadOffset;                           //Minimum High-priority thread offset supported by node.
    k32s maxHighThreadOffset;                           //Maximum High-priority thread offset supported by node.

    kTempProbeId mainProbe;                             //Specifies stamp temperature source (internal or external).
    kBool recoveryEnabled;                              //Is automatic error recovery enabled?
    k64u healthLogPeriod;                               //Health logging period (microseconds).

    kDaughterboardModel daughterboardModel;             //Daughterboard model.

    kArrayList eventManagers;                           //Event managers -- kArrayList<kEventManager>.
    kArrayList cameras;                                 //Camera modules -- kArrayList<kCamera>.
    kArrayList lights;                                  //Light modules -- kArrayList<kLight>.
    kArrayList projectors;                              //Projector modules -- kArrayList<kProjector>.
    kArrayList analogOutputs;                           //Analog output modules -- kArrayList<kAnalogOut>.
    kArrayList digitalOutputs;                          //Digital output modules -- kArrayList<kDigitalOut>.
    kArrayList serialOutputs;                           //Serial output modules -- kArrayList<kSerialOut>.
    kArrayList ioTests;                                 //I/O test modules -- kArrayList<kIoTest>.
    kArrayList gpioBanks;                               //GPIO banks -- kArrayList<kGpioBank>.

    kOrientation orientation;                           //Orientation module (optional -- can be null).
    kEncoder encoder;                                   //Encoder module (optional -- can be null). 
    kTestJig testJig;                                   //Test jig module (optional -- can be null). 
    kTempControl tempControl;                           //Temp control module (optional -- can be null). 
    kPipe pipe;                                         //Pipe module.
    kPipe actions;                                      //Actions module.
    kNet net;                                           //Net module.
    kStorage storage;                                   //Storage module.

    k64u powerSaverTimeout;                             //Power saver timeout.
    k64u powerSaverThreshold;                           //Power saver threshold.
    kBool lightDriverHighPowerEnabled;                  //Is laser driver high power enabled?

    k64u cameraDataBandwidth;                          //Camera data bandwidth.
} kNodeClass; 

kDeclareClassEx(kFs, kNode, kObject)
        
kFsFx(kStatus) kNode_Construct(kNode* node, kSystem system, kDiscoveryProvider discoveryProvider, const kDiscoveryInfo* info, kAlloc allocator); 

kFsFx(kStatus) kNode_Init(kNode node, kType type, kSystem system, kDiscoveryProvider discoveryProvider, const kDiscoveryInfo* info, kAlloc alloc); 
kFsFx(kStatus) kNode_VRelease(kNode node); 

kFsFx(kStatus) kNode_InvalidateDiscovery(kNode node); 
kFsFx(kStatus) kNode_BeginDiscoveryCycle(kNode node); 
kFsFx(kStatus) kNode_SetDiscovered(kNode node, const kDiscoveryInfo* info); 
kFsFx(kStatus) kNode_NoteAddressChange(kNode node);
kFsFx(kDiscoveryProvider) kNode_DiscoveryProvider(kNode node); 
kFsFx(const kDiscoveryInfo*) kNode_DiscoveryInfo(kNode node); 

kFsFx(kStatus) kNode_ConstructProvider(kNode node, kNodeProvider* provider); 

kFsFx(kStatus) kNode_OnDiscoveryChangeAddress(kNode node, kSize nodeInterfaceIndex, const kIpConfig* ipConfig);

kFsFx(kStatus) kNode_OnProviderError(kNode node, kObject sender, kPointer args); 
kFsFx(kStatus) kNode_OnCancelQuery(kNode node, kObject sender, kPointer args); 

kFsFx(kStatus) kNode_BeginAttemptConnect(kNode node); 
kFsFx(kStatus) kNode_EndAttemptConnect(kNode node); 

kFsFx(kStatus) kNode_BeginAttemptOpenHealth(kNode node); 
kFsFx(kStatus) kNode_EndAttemptOpenHealth(kNode node); 

kFsFx(kStatus) kNode_BeginAttemptOpenData(kNode node); 
kFsFx(kStatus) kNode_EndAttemptOpenData(kNode node); 

kFsFx(kStatus) kNode_ResetEx(kNode node, kBool restartApp, kBool defer = kFALSE); 

kFsFx(kStatus) kNode_BeginReset(kNode node); 
kFsFx(kStatus) kNode_EndReset(kNode node); 
kFsFx(kBool) kNode_ResetCompleted(kNode node); 
kFsFx(kStatus) kNode_ClearReset(kNode node); 

kFsFx(kStatus) kNode_OnStatusCheck(kNode node, kSystem system, kPointer unused); 

kFsFx(kStatus) kNode_OnData(kNode node, kObject sender, kMsgInfo message); 
kFsFx(kStatus) kNode_OnHealth(kNode node, kObject sender, kObject info); 

kFsFx(kStatus) kNode_InvalidateDataSession(kNode node);

kFsFx(kStatus) kNode_SetDeviceConfigModified(kNode node);
kFsFx(kStatus) kNode_SetModeConfigModified(kNode node);

kFsFx(kStatus) kNode_SetDeviceValue(kNode node, kType type, void* variableRef, kSize variableSize, const void* valueRef, kSize valueSize); 
kFsFx(kStatus) kNode_SetDeviceText(kNode node, kChar* variable, kSize capacity, const kChar* input); 
kFsFx(kStatus) kNode_SetModeValue(kNode node, kType type, void* variableRef, kSize variableSize, const void* valueRef, kSize valueSize); 
kFsFx(kStatus) kNode_SetModeText(kNode node, kChar* variable, kSize capacity, const kChar* input); 

template <typename T, typename V>
kStatus kNode_SetModeValue(kNode node, T& variable, T value, kStatus(kCall* infoFx)(kPointer context, V* info), kPointer context, kAdjust adjustment)
{
    if (adjustment != kADJUST_NONE)
    {
        V info; 

        infoFx(context, &info); 

        value = xkAdjustInfoValue(info, value, adjustment);
    }

    if (variable != value)
    {
        variable = value;

        if (!kIsNull(node))
        {
            kNode_SetModeConfigModified(node); 
        }
    }

    return kOK; 
}

//the purpose of this method is to resynchronize with remote state after making a change
//that has the side-effect of updating remote configuration (e.g., sending analog output
//calibration data updates mode configuration)
//
//this method will not invalidate existing handles without cause.    
kFsFx(kStatus) kNode_Sync(kNode node, k64u expectedConfigRevision); 

kFsFx(kStatus) kNode_ConditionallyLoadConfig(kNode node, k64u expectedConfigRevision);
kFsFx(kStatus) kNode_LoadConfig(kNode node); 
kFsFx(kStatus) kNode_LoadInfo(kNode node); 

kFsFx(kStatus) kNode_CommitEx(kNode node, kBool commitDeviceConfig, kBool commitModeConfig);
kFsFx(kStatus) kNode_ClearSelfEx(kNode node, kNodeClearOption options); 

kFsFx(kStatus) kNode_VerifySelf(kNode node); 

kFsFx(kStatus) kNode_AppendErrors(kNode node, kArrayList errors); 
kFsFx(kStatus) kNode_AppendError(kNode node, const kChar* error); 
kFsFx(kStatus) kNode_AppendWarnings(kNode node, kArrayList warnings);
kFsFx(kStatus) kNode_AppendWarning(kNode node, const kChar* warning); 
kFsFx(kStatus) kNode_AppendCustomError(kNode node, const kChar* source, const kChar* error); 
kFsFx(kStatus) kNode_AppendCustomWarning(kNode node, const kChar* source, const kChar* warning); 

kFsFx(kStatus) kNode_VerifyInfo(kNode node, const kChar* source, const kChar* field, const kInfo* info); 
kFsFx(kStatus) kNode_VerifyInfo32u(kNode node, const kChar* source, const kChar* field, const kInfo32u* info, k32u value); 
kFsFx(kStatus) kNode_VerifyInfo32s(kNode node, const kChar* source, const kChar* field, const kInfo32s* info, k32s value); 
kFsFx(kStatus) kNode_VerifyInfo64u(kNode node, const kChar* source, const kChar* field, const kInfo64u* info, k64u value); 
kFsFx(kStatus) kNode_VerifyInfo64s(kNode node, const kChar* source, const kChar* field, const kInfo64s* info, k64s value); 
kFsFx(kStatus) kNode_VerifyInfoSize(kNode node, const kChar* source, const kChar* field, const kInfoSize* info, kSize value); 
kFsFx(kStatus) kNode_VerifyInfo64f(kNode node, const kChar* source, const kChar* field, const kInfo64f* info, k64f value); 
kFsFx(kStatus) kNode_VerifyInfoBool(kNode node, const kChar* source, const kChar* field, const kInfoBool* info, kBool value); 
kFsFx(kStatus) kNode_VerifyInfoBits(kNode node, const kChar* source, const kChar* field, const kInfoBits* info, k32u value); 
kFsFx(kStatus) kNode_VerifyIpConfig(kNode node, const kChar* source, const kIpConfig* ipConfig); 
kFsFx(kStatus) kNode_VerifyCameraThroughput(kNode node, const kChar* source); 

kFsFx(kStatus) kNode_ParseDeviceConfig(kNode node, kXml config);
kFsFx(kStatus) kNode_FormatDeviceConfig(kNode node, kXml config); 

kFsFx(kStatus) kNode_ParseModeConfig(kNode node, kXml config); 
kFsFx(kStatus) kNode_ParseModuleList(kNode node, kXml xml, kXmlItem item, kArrayList moduleList, kNodeConstructModuleFx constructFx, kNodeParseModuleFx parseFx); 
kFsFx(kStatus) kNode_ParseModule(kNode node, kXml xml, kXmlItem item, kObject* module, kNodeConstructModuleFx constructFx, kNodeParseModuleFx parseFx); 

kFsFx(kStatus) kNode_GetDeviceConfig(kNode node, kString devCfg);
kFsFx(kStatus) kNode_GetModeConfig(kNode node, kString modeCfg);

kFsFx(kStatus) kNode_FormatModeConfig(kNode node, kXml config); 
kFsFx(kStatus) kNode_FormatModuleList(kNode node, kXml xml, kXmlItem item, const kChar* listName, const kChar* itemName, kArrayList moduleList, kNodeFormatModuleFx formatFx); 
kFsFx(kStatus) kNode_FormatModule(kNode node, kXml xml, kXmlItem item, const kChar* itemName, kObject module, kNodeFormatModuleFx formatFx); 

kFsFx(kStatus) kNode_CreateVirtualDirectory(kNode node, const kChar* parentPath, kChar* nodePath, kSize nodePathCapacity);
kFsFx(kStatus) kNode_WriteVirtualHardwareConfig(kNode node, const kChar* nodePath); 
kFsFx(kStatus) kNode_FormatVirtualHardwareConfig(kNode node, kXml xml, kXmlItem root); 
kFsFx(kStatus) kNode_WriteVirtualContent(kNode node, const kChar* nodePath); 
kFsFx(kStatus) kNode_AdjustVirtualConfig(kNode node, const kChar* nodePath); 

kFsFx(kStatus) kNode_InvalidateConfigRevision(kNode node); 
kFsFx(k64u) kNode_ConfigRevision(kNode node);
kFsFx(kStatus) kNode_SetConfigRevision(kNode node, k64u revision); 

kFsFx(kStatus) kNode_ValidateBlockConfig(kNode node, kXml config); 

kFsFx(kStatus) kNode_ValidateActionConfig(kNode node, kXml config); 
kFsFx(kStatus) kNode_InvokeAction(kNode node, kBool apply, const kChar* typeName, kXml config, kObject input, kObject* output, kAlloc alloc); 
kFsFx(kStatus) kNode_InvokeNamedAction(kNode node, kBool apply, const kChar* name, kObject input, kObject* output, kAlloc alloc); 

kFsFx(k64u) kNode_MinimumEventPeriod(kNode node, kSize eventId); 
kFsFx(k64f) kNode_MaximumCameraDutyCycle(kNode node, kSize cameraId); 
kFsFx(k64u) kNode_MinPatternDuration(kNode node);
kFsFx(k64u) kNode_PlCameraProcessingTime(kNode node, kSize cameraStateId, k64u cameraPeriod);

kFsFx(k64u) kNode_CoupledCameraLeadIn(kNode node, kCamera camera); 
kFsFx(k64u) kNode_CoupledCameraLeadOut(kNode node, kCamera camera); 
kFsFx(k64u) kNode_CoupledLightLeadIn(kNode node, kLight light);
kFsFx(k64u) kNode_CoupledLightLeadOut(kNode node, kLight light);

kFsFx(kSize) kNode_LocalDigitalInputCount(kNode node); 
kFsFx(kSize) kNode_NetworkDigitalInputCount(kNode node); 

kFsFx(kStatus) kNode_EstablishCompatibility(kNode node); 
kFsFx(kStatus) kNode_SetIncompatible(kNode node); 

kFsFx(kStatus) kNode_UpdateRunState(kNode node); 
kFsFx(kStatus) kNode_SetRunState(kNode node, kNodeState runState); 
kFsFx(kNodeState) kNode_State(kNode node); 

kFsFx(kBool) kNode_IsDiscoveryOnline(kNode node); 
kFsFx(kBool) kNode_HasAddressChange(kNode node);
kFsFx(kBool) kNode_HasActivity(kNode node); 
kFsFx(kBool) kNode_HasCommunicationError(kNode node); 
kFsFx(kBool) kNode_IsInconsistent(kNode node); 
kFsFx(kBool) kNode_IsConnected(kNode node); 
kFsFx(kBool) kNode_IsCompatible(kNode node); 
kFsFx(kBool) kNode_IsConnectable(kNode node); 
kFsFx(kBool) kNode_IsReachable(kNode node);
kFsFx(kBool) kNode_IsResponsive(kNode node); 
kFsFx(kBool) kNode_IsReadable(kNode node); 
kFsFx(kBool) kNode_IsConfigurable(kNode node); 
kFsFx(kBool) kNode_IsReady(kNode node); 
kFsFx(kBool) kNode_IsPaused(kNode node); 
kFsFx(kBool) kNode_IsRunning(kNode node); 
kFsFx(kBool) kNode_IsNormal(kNode node); 
kFsFx(kBool) kNode_IsReplay(kNode node); 
kFsFx(kBool) kNode_IsCancelled(kNode node); 
kFsFx(kBool) kNode_ShouldRefresh(kNode node); 
kFsFx(kNodeState) kNode_RunState(kNode node); 

kFsFx(kBool) kNode_IsStation(kNode node);
kFsFx(kBool) kNode_IsLocal(kNode node);

kFsFx(kSystem) kNode_System(kNode node);
kFsFx(kNodeProvider) kNode_Provider(kNode node);

kFsFx(k32u) kNode_MainPlConfig(kNode node);
kFsFx(kVersion) kNode_MainPlVersion(kNode node);

kFsFx(k32u) xkNode_SecondaryPlConfig(kNode node);
kFsFx(kVersion) xkNode_SecondaryPlVersion(kNode node);

kFsFx(kSocType) xkNode_SocType(kNode node);
kFsFx(kControllerType) xkNode_ControllerType(kNode node);
kFsFx(kVersion) xkNode_ControllerVersion(kNode node);

kFsFx(k32u) kNode_ProcessorCount(kNode node);

kFsFx(kBool) kNode_HasThreadAffinity(kNode node);
kFsFx(kThreadPriorityClass) kNode_MinThreadPriorityClass(kNode node);
kFsFx(kThreadPriorityClass) kNode_MaxThreadPriorityClass(kNode node);
kFsFx(k32s) kNode_MinThreadPriorityOffset(kNode node, kThreadPriorityClass priorityClass);
kFsFx(k32s) kNode_MaxThreadPriorityOffset(kNode node, kThreadPriorityClass priorityClass);

/** 
 * Begins to access SPI bus.
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   deviceType      Device type of SPI bus.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_BeginSpi(kNode node, kSpiDeviceType deviceType);

/** 
 * Ends SPI bus access.
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_EndSpi(kNode node);

/** 
 * Reads data from SPI bus.
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   opCode          Operation code.
 * @param   opSize          Number of bytes to write operation code.
 * @param   data            Data read from SPI.
 * @param   dataSize        Number of bytes to read.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_SpiRead(kNode node, kByte* opCode, kSize opSize, kByte* data, kSize dataSize);

/** 
 * Writes data to SPI bus.
 * 
 * @public                  @memberof kNode
 * @param   node            Node object.
 * @param   opCode          Operation code.
 * @param   opSize          Number of bytes to write operation code.
 * @param   data            Data to write.
 * @param   dataSize        Number of bytes to write.
 * @return                  Operation status. 
 */
kFsFx(kStatus) kNode_SpiWrite(kNode node, kByte* opCode, kSize opSize, kByte* data, kSize dataSize);


/** 
* [Deprecated] replaced with kNode_SetLedMode
* 
* Enables or disables the specified node indicator LED. 
* 
* @public              @memberof kNode
* @param   node        Node object. 
* @param   instance    LED instance.
* @param   enabled     LED state.
* @return              Operation status. 
*/
kFsFx(kStatus) kNode_EnableLed(kNode node, kLed instance, kBool enabled);

//can be used to share non-authoritative time information with a remote node; if the node has
//has already been configured by an authoritative source, it will disregard this information
kFsFx(kStatus) kNode_ShareDateTime(kNode node, kDateTime dateTime); 

kFsFx(kStatus) kNode_FormatNodeInfo(kNode node, kString info);

#define kNode_SetDeviceValueT(NODE, TYPE_SYMBOL, VAR, VAL)             \
    kNode_SetDeviceValue(NODE, kTypeOf(TYPE_SYMBOL), &(VAR), sizeof(VAR), &(VAL), sizeof(VAL))

#define kNode_SetModeValueT(NODE, TYPE_SYMBOL, VAR, VAL)             \
    kNode_SetModeValue(NODE, kTypeOf(TYPE_SYMBOL), &(VAR), sizeof(VAR), &(VAL), sizeof(VAL))

kFsFx(kStatus) xkNode_GenerateReportImpl(kNode node, kString nodeReport);

//[Deprecated] Update code to use kNode_Report instead.
kInlineFx(kStatus) kNode_GenerateReport(kNode node, kString nodeReport)
{
    return xkNode_GenerateReportImpl(node, nodeReport); 
}
kDeprecate(kNode_GenerateReport)

#endif
