/** 
 * @file    kIoTest.h
 * @brief   Declares the kIoTest type. 
 *
 * @internal
 * Copyright (C) 2014-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_IO_TEST_H
#define K_FIRESYNC_IO_TEST_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kIoTest
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents I/O test configuration.
 */
//typedef kObject kIoTest;        --forward-declared in kFsDef.x.h

/** 
 * Gets the id of the output. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @return              Returns the unique id of this test. 
 */
kFsFx(kSize) kIoTest_Id(kIoTest test);

/** 
 * Resets test output device settings to defaults. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_Clear(kIoTest test);

/** 
 * Reset the I/O test module.
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_Reset(kIoTest test);

/** 
 * Sets serial receiver mode.
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   mode        Serial receiver mode.  
 * @return              Control source id.
 */
kFsFx(kStatus) kIoTest_SetSerialMode(kIoTest test, kIoTestSerialMode mode);

/** 
 * Gets serial receiver mode. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @return              Returns the serial receiver mode. 
 */
kFsFx(kIoTestSerialMode) kIoTest_SerialMode(kIoTest test);

/** 
 * Reports constraint and validity information for the SerialMode setting. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_SerialModeInfo(kIoTest test, kInfoBits* info);

/** 
 * Sets the bit rate for UART. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   rate        Specifies the bit rate.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_SetBitRate(kIoTest test, k32u rate);

/** 
 * Gets the bit rate for UART.
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @return              Bit rate.
 */
kFsFx(k32u) kIoTest_BitRate(kIoTest test);

/** 
 * Reports constraint and validity information for the bit rate setting. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_BitRateInfo(kIoTest test, kInfo32u* info);

/** 
 * Sets Gocator 20x0 master mode. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   masterMode  Gocator 20x0 master mode. 
 * @return              Returns the control mode of the test. 
 */
kFsFx(kStatus) kIoTest_SetG20x0MasterMode(kIoTest test, kBool masterMode);

/** 
 * Gets Gocator 20x0 master mode. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @return              Returns the Gocator 20x0 master mode. 
 */
kFsFx(kBool) kIoTest_G20x0MasterMode(kIoTest test);

/** 
 * Reports constraint and validity information for the G20x0MasterMode setting. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_G20x0MasterModeInfo(kIoTest test, kInfoBool* info);

/** 
 * Sets safety output state. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   safety      Safety output state.  
 * @return              Returns the control mode of the test. 
 */
kFsFx(kStatus) kIoTest_SetSafety(kIoTest test, kBool safety);

/** 
 * Gets safety output state. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @return              Returns the safety output state. 
 */
kFsFx(kBool) kIoTest_Safety(kIoTest test);

/** 
 * Reports constraint and validity information for the Safety setting. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_SafetyInfo(kIoTest test, kInfoBool* info);

/** 
 * Sets digital output state. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   index       Digital output index.  
 * @param   state       Digital output state.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_SetDigitalOut(kIoTest test, kSize index, kBool state);

/** 
 * Gets digital output state. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   index       Digital output index.  
 * @return              Returns the digital output state.  
 */
kFsFx(kBool) kIoTest_DigitalOut(kIoTest test, kSize index);

/** 
 * Reports constraint and validity information for the DigitalOut setting. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   index       Digital output index.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_DigitalOutInfo(kIoTest test, kSize index, kInfoBool* info);

/** 
 * Sets Sync I/O value. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   syncIo      Sync I/O value.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_SetSyncIo(kIoTest test, k32u syncIo);

/** 
 * Gets Sync I/O value. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @return              Returns the sync I/O value. 
 */
kFsFx(k32u) kIoTest_SyncIo(kIoTest test);

/** 
 * Reports constraint and validity information for the SyncIo setting. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_SyncIoInfo(kIoTest test, kInfo32u* info);

/** 
 * Sets ADC moving average window size. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   window      Moving average window size.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_SetMovingAvgWindow(kIoTest test, kSize window);

/** 
 * Gets ADC moving average window size. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @return              Returns the moving average window size. 
 */
kFsFx(kSize) kIoTest_MovingAvgWindow(kIoTest test);

/** 
 * Reports constraint and validity information for the ADC moving average window size. 
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_MovingAvgWindowInfo(kIoTest test, kInfoSize* info);

/** 
 * Retrieve statistics.
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   stats       Receives the stats structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_Stats(kIoTest test, kIoTestStats* stats);

/** 
 * Read serial data.
 *
 * @public              @memberof kIoTest
 * @param   test        I/O test object.  
 * @param   buffer      Destination for bytes that are read.
 * @param   capacity    Buffer capacity, in bytes. 
 * @param   bytesRead   Receives count of bytes read. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kIoTest_ReadSerial(kIoTest test, void* buffer, kSize capacity, kSize* bytesRead);

/**
 * External temperature probe AD7415 is not supported in FireSync 6+.
 * @public              @memberof kIoTest
 * @param   node        The device object.
 * @param   deviceId    The device hex value.
 * @param   temp        The external temperature probe value.
 */
kFsFx(kStatus) kIoTest_ReadAd7415(kNode node, k32u deviceId, k32s* temp);

#include <kFireSync/Client/kIoTest.x.h>

#endif
