/** 
 * @file    kGpioBank.h
 * @brief   Declares the kGpioBank type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_GPIO_BANK_H
#define K_FIRESYNC_GPIO_BANK_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kGpioBank
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents a local node's GPIO bank. 
 */
//typedef kObject kGpioBank;        --forward-declared in kFsDef.x.h

/** 
 * Gets the index of the GPIO bank module.
 *
 * @public              @memberof kGpioBank
 * @param   bank        Gpio Bank object.  
 * @return              Returns the index of the Gpio Bank module.
 */
kFsFx(kSize) kGpioBank_Index(kGpioBank bank);

/** 
 * Gets the type of the GPIO bank module.
 *
 * @public              @memberof kGpioBank
 * @param   bank        Gpio Bank object.  
 * @return              Returns the type of the Gpio Bank module.
 */
kFsFx(kGpioBankType) kGpioBank_Type(kGpioBank bank);

/** 
 * Gets the actual number of GPIO pins in this bank.
 *
 * @public              @memberof kGpioBank
 * @param   bank        Gpio Bank object.  
 * @return              Number of GPIO pins. 
 */
kFsFx(kSize) kGpioBank_PinCount(kGpioBank bank);

/** 
 * Sets the GPIO pin type, specified by an index. The pin type determines whether or not input
 * debouncing takes place.
 *
 * @public              @memberof kGpioBank
 * @param   bank        Gpio Bank object.  
 * @param   pinIndex    Gpio pin index.  
 * @param   pinType     Specifies the pin type.
 * @return              Operation status. 
 */
kFsFx(kStatus) kGpioBank_SetPinType(kGpioBank bank, kSize pinIndex, kGpioPin pinType);

/** 
 * Gets the GPIO pin type specified by an index.  The pin type determines whether or not input
 * debouncing takes place.
 *
 * @public              @memberof kGpioBank
 * @param   bank        Gpio Bank object.  
 * @param   pinIndex    Gpio pin index.  
 * @return              Pin type. 
 */
kFsFx(kGpioPin) kGpioBank_PinType(kGpioBank bank, kSize pinIndex);

/** 
 * Enables or disables input events for this GPIO pin. Input events are received asynchronously on the FireSync health channel.
 *
 * @public              @memberof kGpioBank
 * @param   bank        Gpio Bank object.  
 * @param   pinIndex    Gpio pin index.  
 * @param   enabled     Specifies whether events are enabled for the specified Gpio pin.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kGpioBank_EnableEvents(kGpioBank bank, kSize pinIndex, kBool enabled);

/** 
 * Reports whether input events is enabled for this GPIO pin. Input events are received asynchronously on the FireSync health channel.
 *
 * @public              @memberof kGpioBank
 * @param   bank        Gpio Bank object.  
 * @param   pinIndex    Gpio pin index.  
 * @return              Returns whether events are enabled for the specified Gpio pin.
 */
kFsFx(kBool) kGpioBank_EventsEnabled(kGpioBank bank, kSize pinIndex);

/** 
 * Sets the output state for one GPIO pin in this bank, specified by an index.
 *
 * @public              @memberof kGpioBank
 * @param   bank        Gpio Bank object.  
 * @param   pinIndex    Gpio pin index.  
 * @param   enabled     Output state.
 * @return              Operation status. 
 */
kFsFx(kStatus) kGpioBank_SetState(kGpioBank bank, kSize pinIndex, kBool enabled);

/** 
 * Sets the output state for one or more GPIO pins in this bank.
 *
 * This function takes 64-bit values and can be used to change the output state of specific pins, while leaving
 * the output state for all the other pins unchanged.
 *
 * The output state can only be changed for output pins. The behavior for changing input pins is undefined.
 *
 * @public              @memberof kGpioBank
 * @param   bank        Gpio Bank object.  
 * @param   mask        Bitset to specify which pin(s) should be changed.  
 * @param   state       Bitset to specify the output state(s).
 * @return              Operation status. 
 */
kFsFx(kStatus) kGpioBank_SetState64u(kGpioBank bank, k64u mask, k64u state);

/** 
 * Gets the output state for one GPIO pin in this bank, specified by an index.
 *
 * @public              @memberof kGpioBank
 * @param   bank        Gpio Bank object.  
 * @param   pinIndex    Gpio pin index.  
 * @param   enabled     Recieves the output state.
 * @return              Operation status. 
 */
kFsFx(kStatus) kGpioBank_GetState(kGpioBank bank, kSize pinIndex, kBool* enabled);

/** 
 * Gets the output state for all GPIO pins in this bank.
 *
 * This function uses a 64-bit value to represent the output state of all pins in this bank.
 *
 * @public              @memberof kGpioBank
 * @param   bank        Gpio Bank object.  
 * @param   state       Output state bitset.
 * @return              Operation status. 
 */
kFsFx(kStatus) kGpioBank_GetState64u(kGpioBank bank, k64u* state);

#include <kFireSync/Client/kGpioBank.x.h>

#endif
