/** 
 * @file    kDigitalOut.h
 * @brief   Declares the kDigitalOut type. 
 *
 * @internal
 * Copyright (C) 2008-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_DIGITAL_OUT_H
#define K_FIRESYNC_DIGITAL_OUT_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kDigitalOut
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents digital configuration.
 */
//typedef kObject kDigitalOut;        --forward-declared in kFsDef.x.h

/** 
 * Gets the id of the output. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @return              Returns the unique id of this output. 
 */
kFsFx(kSize) kDigitalOut_Id(kDigitalOut digital);

/** 
 * Resets digital output mode settings to defaults. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_Clear(kDigitalOut digital);

/** 
 * Enables or disables the digital. 
 *
 * Disabled digitals are not checked during verification, and do not respond to 
 * Start/Pause/Resume/Stop commands. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   enable      Specifies whether to enable or disable the digital.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_Enable(kDigitalOut digital, kBool enable);

/** 
 * Reports whether the digital is currently enabled. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @return              Returns whether the digital is currently enabled. 
 */
kFsFx(kBool) kDigitalOut_IsEnabled(kDigitalOut digital);

/** 
 * Sets the control mode of the digital.
 *
 * For control modes that suppport multiple sources (e.g. kDIGITAL_OUT_CONTROL_TRIGGERED_BY_EVENT), 
 * use the kDigitalOut_SetControlId function to specify a specific source.
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   type        Digital control mode.
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_SetControl(kDigitalOut digital, kDigitalOutControl type);

/** 
 * Reports the control mode of the digital. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @return              Returns the control mode of the digital. 
 */
kFsFx(kDigitalOutControl) kDigitalOut_Control(kDigitalOut digital);

/** 
 * Reports constraint and validity information for the Control setting. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_ControlInfo(kDigitalOut digital, kInfoBits* info);

/** 
 * Sets the control source id for a digital. 
 *
 * For control modes that suppport multiple sources (e.g. kDIGITAL_OUT_CONTROL_TRIGGERED_BY_EVENT), 
 * this function cam be used to specify a specific source id.
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   id          Specifies the control source id.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_SetControlId(kDigitalOut digital, k32u id);

/** 
 * Gets the control source id for a digital.
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @return              Control source id.
 */
kFsFx(k32u) kDigitalOut_ControlId(kDigitalOut digital);

/** 
 * Reports constraint and validity information for the ControlId setting. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_ControlIdInfo(kDigitalOut digital, kInfo* info);

/** 
 * Sets the polarity for a digital. 
 *
 * @public               @memberof kDigitalOut
 * @param   digital      Digital object.  
 * @param   isActiveHigh Specifies the polarity.  
 * @return               Operation status. 
 */
kFsFx(kStatus) kDigitalOut_SetPolarity(kDigitalOut digital, kBool isActiveHigh);

/** 
 * Gets the polarity for a digital.
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @return              kTRUE if active high. 
 */
kFsFx(kBool) kDigitalOut_Polarity(kDigitalOut digital);

/** 
 * Reports constraint and validity information for the Polarity setting. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_PolarityInfo(kDigitalOut digital, kInfoBool* info);

/** 
 * Sets the delay for the digital, in nanoseconds.
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   delay       Delay value.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_SetDelay(kDigitalOut digital, k64u delay, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the delay for the digital, in nanoseconds.  
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @return              Returns the delay for the digital, in nanoseconds.  
 */
kFsFx(k64u) kDigitalOut_Delay(kDigitalOut digital);

/** 
 * Reports constraint and validity information for the Delay setting. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_DelayInfo(kDigitalOut digital, kInfo64u* info);

/** 
 * Sets the exposure for the digital, in nanoseconds.
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   exposure    Exposure value.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_SetExposure(kDigitalOut digital, k64u exposure, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the exposure for the digital, in nanoseconds.  
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @return              Returns the exposure for the digital, in nanoseconds.  
 */
kFsFx(k64u) kDigitalOut_Exposure(kDigitalOut digital);

/** 
 * Reports constraint and validity information for the Exposure setting. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_ExposureInfo(kDigitalOut digital, kInfo64u* info);

/** 
 * Sets the gap for the digital, in nanoseconds.
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   gap         Gap value.   
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_SetGap(kDigitalOut digital, k64u gap, kAdjust adjustment = kADJUST_NONE);

/** 
 * Gets the gap for the digital, in nanoseconds.  
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @return              Returns the gap for the digital, in nanoseconds.  
 */
kFsFx(k64u) kDigitalOut_Gap(kDigitalOut digital);

/** 
 * Reports constraint and validity information for the Gap setting. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_GapInfo(kDigitalOut digital, kInfo64u* info);

/** 
 * Reports the time required by this digital output to respond to an event. 
 *
 * Minimum period values are specified in FireSync nanoseconds (~ 1 ns).
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @return              Minimum period, in nanoseconds. 
 */
kFsFx(k64u) kDigitalOut_MinimumPeriod(kDigitalOut digital);

/** 
 * Sets the output state.
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   state       Output value.   
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_SetState(kDigitalOut digital, kBool state);

/** 
 * Retrieve statistics.
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @param   stats       Receives the stats structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_Stats(kDigitalOut digital, kDigitalOutStats* stats);

/** 
 * Provides a software trigger to a digital output. 
 *
 * @public              @memberof kDigitalOut
 * @param   digital     Digital object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kDigitalOut_Trigger(kDigitalOut digital);

#include <kFireSync/Client/kDigitalOut.x.h>

#endif
