/** 
 * @file    kActions.h
 * @brief   Declares the kActions class. 
 *
 * @internal
 * Copyright (C) 2014-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ACTIONS_H
#define K_FIRESYNC_ACTIONS_H

#include <kFireSync/kNodeDef.h>

/**
 * @class   kActions
 * @extends kObject
 * @ingroup kFireSync-Client
 * @brief   Represents configuration for a collection of actions.
 */
//typedef kObject kActions;        --forward-declared in kFsDef.x.h

/** 
 * Gets the number of action types currently available on this node. 
 *
 * The action types available on a particular node depend on the firmware application currently 
 * installed on the node.  
 *
 * @public              @memberof kActions
 * @param   actions     Actions object.  
 * @return              Returns the number of action types currently available on this node. 
 */
kFsFx(kSize) kActions_TypeCount(kActions actions); 

/** 
 * Gets the name of the action type at the specified index.
 *
 * @public              @memberof kActions
 * @param   actions     Actions object.  
 * @param   index       Index of the action type.   
 * @return              Action type name. 
 */
kFsFx(const kChar*) kActions_TypeNameAt(kActions actions, kSize index); 

/** 
 * Gets a description of the action type at the specified index.
 *
 * @public              @memberof kActions
 * @param   actions     Actions object.  
 * @param   index       Index of the action type.   
 * @return              Action type description.
 */
kFsFx(const kChar*) kActions_TypeDescriptionAt(kActions actions, kSize index); 

/** 
 * Resets actions collection settings to defaults.
 *
 * @public                      @memberof kActions
 * @param   actions             Actions object.  
 * @return                      Operation status. 
 */
kFsFx(kStatus) kActions_Clear(kActions actions); 

/** 
 * Creates and adds a new action to the actions collection.
 *
 * The new action is added to the end of the actions collection. 
 * 
 * This function involves communication with the remote node. 
 *
 * @public              @memberof kActions
 * @param   actions     Actions object.  
 * @param   typeName    Specifies the type of action to add.   
 * @param   name        Name of the new action.  
 * @param   action      Receives the new action object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kActions_AddAction(kActions actions, const kChar* typeName, const kChar* name, kAction* action); 

/** 
 * Destroys the specified action and removes it from the actions collection.
 *
 * @public              @memberof kActions
 * @param   actions     Actions object.  
 * @param   action      Action object to be deleted.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kActions_DeleteAction(kActions actions, kAction action); 

/** 
 * Finds an existing action by name. 
 *
 * @public              @memberof kActions
 * @param   actions     Actions object.  
 * @param   name        Name of the action.  
 * @param   action      Receives the action object, if found. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kActions_FindAction(kActions actions, const kChar* name, kAction* action); 

/** 
 * Gets the total number of actions belonging to the actions collection.
 *
 * @public              @memberof kActions
 * @param   actions     Actions object.  
 * @return              Total number of actions belonging to the actions collection. 
 */
kFsFx(kSize) kActions_ActionCount(kActions actions); 

/** 
 * Gets the action at the specified index.
 *
 * @public              @memberof kActions
 * @param   actions     Actions object.  
 * @param   index       Index of the action.  
 * @return              Action at the specified index.
 */
kFsFx(kAction) kActions_ActionAt(kActions actions, kSize index); 

/** 
 * Invokes an action of the given type without applying the current settings to this node first. 
 * 
 * This method allows an action to be invoked without first adding the action to the 
 * actions collection.  
 *
 * If the action changes remote configuration, kNode_Refresh can optionally be called to 
 * refresh local configuration.
 * 
 * @public              @memberof kActions
 * @param   actions     Actions object.  
 * @param   typeName    Action type name. 
 * @param   config      Action configuration XML (can be kNULL). 
 * @param   input       Action input object (can be kNULL). 
 * @param   output      Receives action output object (can be kNULL). 
 * @param   alloc       Memory allocator for action output object (or kNULL for default allocator). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kActions_Invoke(kActions actions, const kChar* typeName, kXml config, kObject input, kObject* output, kAlloc alloc); 

/** 
 * Applies current setting to the node and invokes an action of the given type. 
 * 
 * This method allows an action to be invoked without first adding the action to the 
 * actions collection.  
 *
 * Local configuration is automatically refreshed afterwards so no call to kNode_Refresh is necessary.
 * 
 * @public              @memberof kActions
 * @param   actions     Actions object.  
 * @param   typeName    Action type name. 
 * @param   config      Action configuration XML (can be kNULL). 
 * @param   input       Action input object (can be kNULL). 
 * @param   output      Receives action output object (can be kNULL). 
 * @param   alloc       Memory allocator for action output object (or kNULL for default allocator). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kActions_InvokeAndSync(kActions actions, const kChar* typeName, kXml config, kObject input, kObject* output, kAlloc alloc);

#include <kFireSync/Client/kActions.x.h>

#endif
