/** 
 * @file    kProjectorAutoInt.h
 * @brief   Declares the kProjectorAutoInt type. 
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_PROJECTOR_AUTO_INT_H
#define K_FIRESYNC_PROJECTOR_AUTO_INT_H

#include <kFireSync/Client/Projector/Extensions/kProjectorExt.h>

/**
 * @class   kProjectorAutoInt
 * @extends kProjectorAutoIntExt
 * @ingroup kFireSync-Client-Projector-Extensions
 * @brief   Represents parameters for the projector auto-intensity extension.
 */
//typedef kProjectorExt kProjectorAutoInt;            --forward-declared in kFsDef.x.h

/**
 * Sets the minimum intensity defined by this extension.
 *
 * @public              @memberof kProjectorAutoInt
 * @param   extension   Projector object.
 * @param   intensity   Desired minimum intensity.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status.
 */
kFsFx(kStatus) kProjectorAutoInt_SetMinIntensity(kProjectorAutoInt extension, k32u intensity, kAdjust adjustment = kADJUST_NONE);

/**
 * Reports the minimum intensity defined by this extension.
 *
 * @public              @memberof kProjectorAutoInt
 * @param   extension   Projector object.
 * @return              Minimum intensity.
 */
kFsFx(k32u) kProjectorAutoInt_MinIntensity(kProjectorAutoInt extension);

/**
* Reports constraint and validity information for the minimum intensity.
*
* @public              @memberof kProjectorAutoInt
* @param   extension   Projector object.
* @param   info        Receives the info structure.
* @return              Operation status.
*/
kFsFx(kStatus) kProjectorAutoInt_MinIntensityInfo(kProjectorAutoInt extension, kInfo32u* info);

/**
 * Sets the maximum intensity defined by this extension.
 *
 * @public              @memberof kProjectorAutoInt
 * @param   extension   Projector object.
 * @param   intensity   Desired minimum intensity.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status.
 */
kFsFx(kStatus) kProjectorAutoInt_SetMaxIntensity(kProjectorAutoInt extension, k32u intensity, kAdjust adjustment = kADJUST_NONE);

/**
 * Reports the maximum intensity defined by this extension.
 *
 * @public              @memberof kProjectorAutoInt
 * @param   extension   Projector object.
 * @return              Maximum intensity.
 */
kFsFx(k32u) kProjectorAutoInt_MaxIntensity(kProjectorAutoInt extension);

/**
 * Reports constraint and validity information for the maximum intensity.
 *
 * @public              @memberof kProjectorAutoInt
 * @param   extension   Projector object.
 * @param   info        Receives the info structure.
 * @return              Operation status.
 */
kFsFx(kStatus) kProjectorAutoInt_MaxIntensityInfo(kProjectorAutoInt extension, kInfo32u* info);

/**
 * Reports the number of maximum entries allowed for this extension.
 *
 * @public              @memberof kProjectorAutoInt
 * @param   extension   Projector object.
 * @return              Entry count.
 */
kFsFx(kSize) kProjectorAutoInt_EntryCount(kProjectorAutoInt extension);

/**
 * Sets the slope with units exposure (us) per intensity count.
 *
 * @public              @memberof kProjectorAutoInt
 * @param   extension   Projector object.
 * @param   entryIndex  Slope index.
 * @param   slope       Slope value.
 * @return              Operation status.
 */
kFsFx(kStatus) kProjectorAutoInt_SetSlope(kProjectorAutoInt extension, kSize entryIndex, k64f slope);

/**
 * Reports the current slope used for auto intensity.
 *
 * @public              @memberof kProjectorAutoInt
 * @param   extension   Projector object.
 * @param   entryIndex  Slope index.
 * @return              Slope value.
 */
kFsFx(k64f) kProjectorAutoInt_Slope(kProjectorAutoInt extension, kSize entryIndex);

/**
 * Reports constraint and validity information for the slope values.
 *
 * @public              @memberof kProjectorAutoInt
 * @param   extension   Projector object.
 * @param   entryIndex  Slope index.
 * @param   info        Receives the info structure.
 * @return              Operation status.
*/
kFsFx(kStatus) kProjectorAutoInt_SlopeInfo(kProjectorAutoInt extension, kSize entryIndex, kInfo64f* info);

#include <kFireSync/Client/Projector/Extensions/kProjectorAutoInt.x.h>

#endif
