/** 
 * @file    kNode6Client.h
 * @brief   Declares the kNode6Client class. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_NODE_6_CLIENT_H
#define K_FIRESYNC_NODE_6_CLIENT_H

#include <kFireSync/kNodeDef.h>

/**
 * @internal
 * @class       kNode6Client
 * @extends     kObject
 * @implements  kNodeProvider
 * @ingroup     kFireSync-Client-Network
 * @brief       Provides an implementation of the kNodeProvider interface based on FireSync 6+ protcols.
 */
//typedef kObject kNode6Client;            --forward-declared in kFsDef.x.h 

/** 
 * Constructs a kNode6Client object.
 *
 * @public              @memberof kNode6Client
 * @return              Operation status. 
 */
kFsFx(kStatus) kNode6Client_Construct(kNode6Client* client, kIpAddress address, kAlloc allocator); 

kFsFx(kStatus) kNode6Client_SetControlPort(kNode6Client client, k32u port); 
kFsFx(kStatus) kNode6Client_SetHealthPort(kNode6Client client, k32u port); 
kFsFx(kStatus) kNode6Client_SetDataPort(kNode6Client client, k32u port); 
kFsFx(kStatus) kNode6Client_SetMessageAlloc(kNode6Client client, kAlloc alloc); 

kFsFx(kStatus) kNode6Client_SetErrorHandler(kNode6Client client, kCallbackFx function, kPointer receiver); 
kFsFx(kStatus) kNode6Client_SetCancelHandler(kNode6Client client, kCallbackFx function, kPointer receiver); 
kFsFx(kStatus) kNode6Client_SetProtocolVersion(kNode6Client client, kVersion protocol, kVersion platform); 
kFsFx(kStatus) kNode6Client_BeginOpenControl(kNode6Client client); 
kFsFx(kStatus) kNode6Client_EndOpenControl(kNode6Client client); 
kFsFx(kStatus) kNode6Client_CloseControl(kNode6Client client); 
kFsFx(kStatus) kNode6Client_SetHealthHandler(kNode6Client client, kCallbackFx function, kPointer receiver);
kFsFx(kStatus) kNode6Client_BeginOpenHealth(kNode6Client client); 
kFsFx(kStatus) kNode6Client_EndOpenHealth(kNode6Client client); 
kFsFx(kStatus) kNode6Client_CloseHealth(kNode6Client client); 
kFsFx(kStatus) kNode6Client_SetDataBufferSize(kNode6Client client, kSize size);
kFsFx(kStatus) kNode6Client_SetDataHandler(kNode6Client client, kCallbackFx function, kPointer receiver);
kFsFx(kStatus) kNode6Client_BeginOpenData(kNode6Client client, const k16u* ports, kSize portCount); 
kFsFx(kStatus) kNode6Client_EndOpenData(kNode6Client client); 
kFsFx(kStatus) kNode6Client_EndDataSession(kNode6Client client);
kFsFx(kStatus) kNode6Client_CloseData(kNode6Client client);
kFsFx(kStatus) kNode6Client_InvalidateDataSession(kNode6Client client);

kFsFx(kStatus) kNode6Client_ClearFirmware(kNode6Client client, kNodeFirmwareType type); 
kFsFx(kStatus) kNode6Client_WriteFirmware(kNode6Client client, kNodeFirmwareType type, kStream stream, kSize size, kCallbackFx progress, kPointer context); 
kFsFx(kStatus) kNode6Client_ReloadFirmware(kNode6Client client, kNodeFirmwareType type, kStream stream, kSize size);
kFsFx(kStatus) kNode6Client_Reset(kNode6Client client, kBool restartApp);

kFsFx(kStatus) kNode6Client_WriteFile(kNode6Client client, const kChar* path, kStream stream, kSize size, kCallbackFx progress, kPointer context); 
kFsFx(kStatus) kNode6Client_CopyFile(kNode6Client client, const kChar* sourcePath, const kChar* destPath, kCallbackFx progress, kPointer context); 
kFsFx(kStatus) kNode6Client_ReadFile(kNode6Client client, const kChar* path, kStream stream, kCallbackFx progress, kPointer context); 
kFsFx(kStatus) kNode6Client_DeleteFile(kNode6Client client, const kChar* path); 
kFsFx(kStatus) kNode6Client_GetFileInfo(kNode6Client client, const kChar* path, kStorageItem* item);
kFsFx(kStatus) kNode6Client_CreateDirectory(kNode6Client client, const kChar* path); 
kFsFx(kStatus) kNode6Client_DeleteDirectory(kNode6Client client, const kChar* path); 
kFsFx(kStatus) kNode6Client_ListDirectoryEntries(kNode6Client client, const kChar* path, kBool includeFiles, kBool includeDirectories, kArrayList entries); 
kFsFx(kStatus) kNode6Client_ListDirectoryEntriesEx(kNode6Client client, const kChar* path, kArrayList entries); 

kFsFx(kStatus) kNode6Client_SetDeviceConfig(kNode6Client client, kXml xml, k64u revisionId); 
kFsFx(kStatus) kNode6Client_GetDeviceConfig(kNode6Client client, kXml xml, k64u* revisionId); 
kFsFx(kStatus) kNode6Client_CommitDeviceConfig(kNode6Client client); 
kFsFx(kStatus) kNode6Client_SetModeConfig(kNode6Client client, kXml xml, k64u revisionId); 
kFsFx(kStatus) kNode6Client_GetModeConfig(kNode6Client client, kXml xml, k64u* revisionId); 
kFsFx(kStatus) kNode6Client_SaveModeConfig(kNode6Client client, const kChar* path); 
kFsFx(kStatus) kNode6Client_LoadModeConfig(kNode6Client client, const kChar* path); 
kFsFx(kStatus) kNode6Client_CommitModeConfig(kNode6Client client); 
kFsFx(kStatus) kNode6Client_ValidateBlockConfig(kNode6Client client, kXml config); 

kFsFx(kStatus) kNode6Client_SetDateTime(kNode6Client client, kDateTime dateTime, kBool isAuthoritative); 
kFsFx(kStatus) kNode6Client_IncrementInputCounter(kNode6Client client);

kFsFx(kStatus) kNode6Client_CameraOverride(kNode6Client client, kSize index, const kCameraOverrideItem* overrides, kSize count);
kFsFx(kStatus) kNode6Client_LightOverride(kNode6Client client, kSize index, const kLightOverrideItem* overrides, kSize count);

kFsFx(kStatus) kNode6Client_ListProcedures(kNode6Client client, kArrayList procedures);    
kFsFx(kStatus) kNode6Client_InvokeProcedure(kNode6Client client, const kChar* name, kObject input, kObject* output, kAlloc alloc);  

kFsFx(kStatus) kNode6Client_SetVariable(kNode6Client client, const kChar* name, kObject variable); 
kFsFx(kStatus) kNode6Client_GetVariable(kNode6Client client, const kChar* name, kObject* variable, kAlloc alloc); 
kFsFx(kStatus) kNode6Client_GetVariableInfo(kNode6Client client, const kChar* name, kNodeVariableInfo* info); 
kFsFx(kStatus) kNode6Client_DeleteVariable(kNode6Client client, const kChar* name); 
kFsFx(kStatus) kNode6Client_ListVariables(kNode6Client client, kArrayList variables); 

kFsFx(kStatus) kNode6Client_ValidateActionConfig(kNode6Client client, kXml config); 
kFsFx(kStatus) kNode6Client_InvokeAction(kNode6Client client, const kChar* typeName, kXml config, kObject input, kObject* output, kAlloc alloc); 
kFsFx(kStatus) kNode6Client_InvokeNamedAction(kNode6Client client, const kChar* name, kObject input, kObject* output, kAlloc alloc); 

kFsFx(kStatus) kNode6Client_ReadMemory(kNode6Client client, kNodeMemorySpace space, k64u offset, kSize length, void* data);
kFsFx(kStatus) kNode6Client_WriteMemory(kNode6Client client, kNodeMemorySpace space, k64u offset, kSize length, const void* data);

kFsFx(kStatus) kNode6Client_ReadRegisters(kNode6Client client, kRegisterModule moduleType, kSize moduleIndex, kArrayList registers);
kFsFx(kStatus) kNode6Client_ReadRegisterOverrides(kNode6Client client, kRegisterModule moduleType, kArrayList registers);
kFsFx(kStatus) kNode6Client_WriteRegisterOverrides(kNode6Client client, kRegisterModule moduleType, kArrayList registers);

kFsFx(kStatus) kNode6Client_I2cRead(kNode6Client client, k32u deviceId, k32u address, kByte* data, kSize size);
kFsFx(kStatus) kNode6Client_I2cWrite(kNode6Client client, k32u deviceId, k32u address, const kByte* data, kSize size);

kFsFx(kStatus) kNode6Client_BeginSpi(kNode6Client client, kSpiDeviceType type);
kFsFx(kStatus) kNode6Client_EndSpi(kNode6Client client);
kFsFx(kStatus) kNode6Client_SpiRead(kNode6Client client, kByte* opCode, kSize opSize, kByte* data, kSize dataSize);
kFsFx(kStatus) kNode6Client_SpiWrite(kNode6Client client, kByte* opCode, kSize opSize, kByte* data, kSize dataSize);

kFsFx(kStatus) kNode6Client_ReadLogHistory(kNode6Client client, kArrayList logItems);
kFsFx(kStatus) kNode6Client_ReadCrashLog(kNode6Client client, kString* crashLog, kAlloc allocator);
kFsFx(kStatus) kNode6Client_ClearCrashLog(kNode6Client client);

kFsFx(kStatus) kNode6Client_BeginStart(kNode6Client client); 
kFsFx(kStatus) kNode6Client_EndStart(kNode6Client client); 
kFsFx(kStatus) kNode6Client_BeginStop(kNode6Client client, kBool synchronizeData); 
kFsFx(kStatus) kNode6Client_EndStop(kNode6Client client); 
kFsFx(kStatus) kNode6Client_BeginEngage(kNode6Client client, k64u startTime, k64s startEncoder); 
kFsFx(kStatus) kNode6Client_EndEngage(kNode6Client client); 

kFsFx(kStatus) kNode6Client_BeginStartReplay(kNode6Client client); 
kFsFx(kStatus) kNode6Client_EndStartReplay(kNode6Client client); 
kFsFx(kStatus) kNode6Client_Replay(kNode6Client client, kObject input, kArrayList output); 

kFsFx(kStatus) kNode6Client_ScheduleEvent(kNode6Client client, k32u eventManagerId, k64s target);    
kFsFx(kStatus) kNode6Client_TriggerEvent(kNode6Client client, k32u eventManagerId); 
kFsFx(kStatus) kNode6Client_TriggerCamera(kNode6Client client, kSize index);
kFsFx(kStatus) kNode6Client_TriggerLight(kNode6Client client, kSize index);
kFsFx(kStatus) kNode6Client_SetLightState(kNode6Client client, kSize index, kBool enabled);
kFsFx(kStatus) kNode6Client_SetDigitalOutState(kNode6Client client, kSize index, kBool state);
kFsFx(kStatus) kNode6Client_TriggerDigitalOut(kNode6Client client, kSize index);
kFsFx(kStatus) kNode6Client_EnqueueAnalogOut(kNode6Client client, kSize index, k32u value);
kFsFx(kStatus) kNode6Client_EnqueueAnalogOutCurrent(kNode6Client client, kSize index, k64f current);
kFsFx(kStatus) kNode6Client_WriteAnalogOutCalibration(kNode6Client client, kSize index, kArray2 calibration, k64u revisionId);

kFsFx(kStatus) kNode6Client_WriteSerialOut(kNode6Client client, kSize index, const void* buffer, kSize size);
kFsFx(kStatus) kNode6Client_WriteRamImage(kNode6Client client, kSize index, kSize stateIndex, kSize imageIndex, kImage image);
kFsFx(kStatus) kNode6Client_WritePrnu(kNode6Client client, kSize index, kImage white, kImage black);
kFsFx(kStatus) kNode6Client_WriteFpn(kNode6Client client, kSize index, kImage offsets);
kFsFx(kStatus) kNode6Client_WriteRangeLut(kNode6Client client, kSize index,
                                                            const kRangeLutParams* params, const kRangeLutArray* xArray,
                                                            const kRangeLutArray* zArray, const kRangeLutArray* validArray);
kFsFx(kStatus) kNode6Client_WritePhaseDecoderLut(kNode6Client client, kSize index, kCameraPhaseDecoderLutType type, kArray1 lut);
kFsFx(kStatus) kNode6Client_SetCameraLvdsPower(kNode6Client client, kSize index, k32u power);
kFsFx(kStatus) kNode6Client_GetCameraLvdsPower(kNode6Client client, kSize index, k32u* power);
kFsFx(kStatus) kNode6Client_WriteAccelerationTestData(kNode6Client client, kSize index, kObject data);
kFsFx(kStatus) kNode6Client_ReadAccelerationTestResult(kNode6Client client, kSize index, kObject* data, kAlloc allocator);

kFsFx(kStatus) kNode6Client_LoadPatterns(kNode6Client client, kSize index, kArray2 patterns);
kFsFx(kStatus) kNode6Client_LoadProjectorRowGains(kNode6Client client, kSize index, kArray1 gains);
kFsFx(kStatus) kNode6Client_LoadProjectorColGains(kNode6Client client, kSize index, kArray1 gains);
kFsFx(kStatus) kNode6Client_ResetEncoder(kNode6Client client);
kFsFx(kStatus) kNode6Client_EnableLed(kNode6Client client, kLed instance, kBool enabled);
kFsFx(kStatus) kNode6Client_SetLedMode(kNode6Client client, kLed instance, kLedMode mode);
kFsFx(kStatus) kNode6Client_SetGpioState(kNode6Client client, kSize bankIndex, k64u mask, k64u state);
kFsFx(kStatus) kNode6Client_GetGpioState(kNode6Client client, kSize bankIndex, k64u* state);

kFsFx(kStatus) kNode6Client_GetInfo(kNode6Client client, kNodeInfo* info); 
kFsFx(kStatus) kNode6Client_GetState(kNode6Client client, kNodeStateInfo* info); 
kFsFx(kStatus) kNode6Client_GetHealth(kNode6Client client, kArrayList stats); 
kFsFx(kStatus) kNode6Client_ReadHealthLog(kNode6Client client, kHealthSummary* summary, kAlloc alloc); 
kFsFx(kStatus) kNode6Client_ClearHealthLog(kNode6Client client);  
kFsFx(kStatus) kNode6Client_GetNodeStats(kNode6Client client, kNodeStats* stats);    
kFsFx(kStatus) kNode6Client_GetEventStats(kNode6Client client, k32u eventManagerId, kEventStats* stats);    
kFsFx(kStatus) kNode6Client_GetCameraStats(kNode6Client client, kSize index, kCameraStats* stats);
kFsFx(kStatus) kNode6Client_GetLightStats(kNode6Client client, kSize index, kLightStats* stats);
kFsFx(kStatus) kNode6Client_GetProjectorStats(kNode6Client client, kSize index, kProjectorStats* stats);
kFsFx(kStatus) kNode6Client_GetAnalogOutStats(kNode6Client client, kSize index, kAnalogOutStats* stats);
kFsFx(kStatus) kNode6Client_GetDigitalOutStats(kNode6Client client, kSize index, kDigitalOutStats* stats);
kFsFx(kStatus) kNode6Client_GetSerialOutStats(kNode6Client client, kSize index, kSerialOutStats* stats);

kFsFx(kStatus) kNode6Client_ResetIoTest(kNode6Client client, kSize index);
kFsFx(kStatus) kNode6Client_GetIoTestStats(kNode6Client client, kSize index, kIoTestStats* stats);
kFsFx(kStatus) kNode6Client_ReadIoTestSerialData(kNode6Client client, kSize index, void* buffer, kSize capacity, kSize* bytesRead);

kFsFx(kStatus) kNode6Client_TestJigCommand(kNode6Client client, kTestJigCommand command, k64u data);
kFsFx(kStatus) kNode6Client_GetTestJigStats(kNode6Client client, kTestJigStats* stats);

kFsFx(kStatus) kNode6Client_LockOutLight(kNode6Client client, kSize index, kBool shouldLock);
kFsFx(kStatus) kNode6Client_SetLightDriverControl(kNode6Client client, kSize index, k64u key, kBool enabled);
kFsFx(kStatus) kNode6Client_GetLightDriverControl(kNode6Client client, kSize index, k64u key, kBool* enabled); 
kFsFx(kStatus) kNode6Client_SetLightDriverPower(kNode6Client client, kSize index, k64u key, k32u power, kBool commit);
kFsFx(kStatus) kNode6Client_GetLightDriverPower(kNode6Client client, kSize index, k64u key, k32u* power); 
kFsFx(kStatus) kNode6Client_SetLightDriverCurrentLimit(kNode6Client client, kSize index, k64u key, k32u currentLimit, kBool commit);
kFsFx(kStatus) kNode6Client_GetLightDriverCurrentLimit(kNode6Client client, kSize index, k64u key, k32u* currentLimit);
kFsFx(kStatus) kNode6Client_GetLightDriverRemainingSlotCount(kNode6Client client, kSize index, k64u key, kSize* remainingCount);
kFsFx(kStatus) kNode6Client_SetLightDriverInfo(kNode6Client client, kSize index, k64u key, kLightModel model, kVersion revision, k32u deviceId);
kFsFx(kStatus) kNode6Client_GetLightDriverInfo(kNode6Client client, kSize index, k64u key, kLightModel* model, kVersion* revision, k32u* deviceId);
kFsFx(kStatus) kNode6Client_SetLightDriverCalibration(kNode6Client provider, kSize index, k64u key, kDataTree ldCal, kBool commit);
kFsFx(kStatus) kNode6Client_GetLightDriverCalibration(kNode6Client provider, kSize index, k64u key, kDataTree* ldCal, kAlloc alloc);
kFsFx(kStatus) kNode6Client_ReadLightDriverTemperature(kNode6Client provider, kSize index, k64u key, k32s* temperature);

kFsFx(kStatus) kNode6Client_StartOrientationCalibration(kNode6Client client);
kFsFx(kStatus) kNode6Client_GetOrientationStats(kNode6Client client, kOrientationStats* stats);

#include <kFireSync/Client/Network/kNode6Client.x.h>

#endif
