/** 
 * @file    kLc900xInfo.x.h
 *
 * @internal
 * Copyright (C) 2020-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_LC900X_INFO_X_H
#define K_FIRESYNC_LC900X_INFO_X_H

//TODO: According to the G2509 datasheet, the "frame overhead time" is the delay between end of exposure and frame readout.
//      The next frame's exposure should not begin during this period, which is defined as:
//                       FOT = T_clk_pix * line_unit_length = 10.47ns * 152 = 1591ns (datasheet pg 71 "Frame Rate Calculation")
//      However, this period was observed to be closer to the value specified below. This mismatch was causing camera errors at
//      at higher speeds. 10000ns was sufficient when frame rate was exposure-limited, but 22500ns was needed when readout-limited.
//      This should be investigated further and verified with GPixel.
#define kLC900X_FRAME_READOUT_DELAY_NS          (22500)                 // Approximate observed period after frame overhead time and before frame readout.

//Imager pixel map. See datasheet for details.
#define kLC900X_START_MARGIN_ROWS               (4)                     // Number of readable rows before active area.
#define kLC900X_END_MARGIN_ROWS                 (46)                    // Number of readable rows after active area and before OB rows.
#define kLC900X_OPTICAL_BLACK_ROWS              (12)                    // Number of optical black rows.
#define kLC900X_OPTICAL_BLACK_COLUMNS           (64)                    // Number of optical black columns saved by PL.
#define kLC900X_MAX_HEIGHT                      (2160)                  // Imager maximum active area height (pixels).
#define kLC900X_MIN_HEIGHT                      (1)                     // Imager maximum active area height (pixels).
#define kLC900X_HEIGHT_GRANULARITY              (1)                     // Imager width granularity(pixels).
#define kLC900X_MAX_WIDTH                       (4192)                  // Imager maximum active area width (pixels). Imager supports 4200 pixels but PL needs this to be divisible by 16.
#define kLC900X_MIN_WIDTH                       (16)                    // Imager maximum active area width (pixels).
#define kLC900X_WIDTH_GRANULARITY               (16)                    // Imager width granularity(pixels).
#define kLC900X_LEFT_GRANULARITY                (16)                    // Window x granularity(pixels). 
#define kLC900X_TOP_GRANULARITY                 (1)                     // Window y granularity(pixels).
#define kLC900X_BLACK_LEVEL_MIN_10_BPP          (-8192)                 // Minimum 14-bit 2's complement value.
#define kLC900X_BLACK_LEVEL_MAX_10_BPP          (8191)                  // Maximum 14-bit 2's complement value.
#define kLC900X_BLACK_LEVEL_MIN_8_BPP           (-2048)                 // Converted for 8-bpp use from 10-bpp.
#define kLC900X_BLACK_LEVEL_MAX_8_BPP           (2047)                  // Converted for 8-bpp use from 10-bpp.
#define kLC900X_MIN_KNEE_COUNT                  (0)                     // Multi-Slope minimum knee count.
#define kLC900X_MAX_KNEE_COUNT                  (0)                     // Multi-Slope maximum knee count.
#define kLC900X_KNEE_COUNT_GRANULARITY          (0)                     // Multi-Slope knee count granularity.
#define kLC900X_RLUT_WIDTH                      (2160)                  // Range LUT width.
#define kLC900X_RLUT_HEIGHT                     (4192)                  // Range LUT height.

#define kLC900X_MAX_EXPOSURE_NS                 (1*1000*1000*1000)      // Maximum exposure, in nanoseconds. This value is not an inherent limitation of the imager.

#define kLC900X_MAX_IMAGER_X_SUB                (1)                     // Maximum x-subsampling supported by imager.
#define kLC900X_MAX_IMAGER_Y_SUB                (4)                     // Maximum y-subsampling supported by imager.

#define kLC900X_ANALOG_GAIN_GRANULARITY         (0.25)                  // Analog gain step granularity.
#define kLC900X_ANALOG_GAIN_MIN_STEP            (1)                     // Maximum analog gain step.
#define kLC900X_ANALOG_GAIN_MAX_STEP            (21)                    // Maximum analog gain step.
#define kLC900X_ANALOG_GAIN_DEFAULT             (4.0f)                  // Default analog gain fraction.
#define kLC900X_DIGITAL_GAIN_GRANULARITY        (0.0078125)             // Digital gain step granularity.
#define kLC900X_DIGITAL_GAIN_MIN_STEP           (0)                     // Minimum digital gain step.
#define kLC900X_DIGITAL_GAIN_MAX_STEP           (1023)                  // Maximum digital gain step.
#define kLC900X_DIGITAL_GAIN_DEFAULT            (1.0f)                  // Default digital gain fraction.

#define kLC900X_DEFAULT_CENTROID_STEP_8K        (32)                    // Range LUT Default CG step (8K entries)
#define kLC900X_DEFAULT_CENTROID_STEP_16K       (16)                    // Range LUT Default CG step (16K entries)
#define kLC900X_DEFAULT_VALID_STEP              (8)                     // Range LUT Default valid step
#define kLC900X_DEFAULT_SLICE_STEP              (16)                    // Range LUT Default slice step

#define kLC900X_POST_EXPOSURE_PIX_CYCLES             (119)                   // Lead out period (pixel clock cycles).
#define kLC900X_32_CH_ROW_COEFFICIENT           (1)                     // Row coefficient for 32-channel configuration (see datasheet Table 37)

#define kLC900X_ROW_UNIT_LENGTH                 (152)                   // This parameter is configurable by the imager, but should not change from this value. Required for timing calculations.
#define kLC900X_PIXEL_CLOCK_MHZ                 (96.0)                  // Imager pixel clock. Different from PL pixel clock. Required for timing calculations.

typedef struct kLc900xInfoClass
{
    kCameraInfoClass base;
} kLc900xInfoClass;

kDeclareClassEx(kFs, kLc900xInfo, kCameraInfo)

kFsFx(kStatus) kLc900xInfo_Init(kLc900xInfo info, kType type, kCameraModel model, kCameraCapability capabilities, kBool isTrailingRequired, kSize frameSizeGranularity, kVersion plVersion, kAlloc alloc);
kFsFx(kStatus) kLc900xInfo_VRelease(kLc900xInfo info);

kFsFx(kBool) kLc900xInfo_VHasGainChannel(kCameraInfo info, kCameraGain channel);
kFsFx(k32u) kLc900xInfo_VMinGainStep(kCameraInfo info, kCameraGain channel);
kFsFx(k32u) kLc900xInfo_VMaxGainStep(kCameraInfo info, kCameraGain channel);
kFsFx(k32u) kLc900xInfo_VGainStep(kLc900xInfo info, kCameraGain channel, k64f gain);
kFsFx(k64f) kLc900xInfo_VGainFactor(kLc900xInfo info, kCameraGain channel, k32u index);
kFsFx(k64f) kLc900xInfo_VDefaultGainFactor(kCameraInfo info, kCameraGain channel);

kFsFx(k64f) kLc900xInfo_VReadoutTime(kLc900xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc900xInfo_VMinInterExposureGap(kLc900xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc900xInfo_VMinInterReadoutGap(kLc900xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc900xInfo_VMinExposure(kLc900xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc900xInfo_VMaxExposure(kLc900xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc900xInfo_VPreExposure(kLc900xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc900xInfo_VPostExposure(kLc900xInfo info, const kCameraInfoParams* params);

kFsFx(k32u) kLc900xInfo_AnalogGainStep(kLc900xInfo info, k64f gain);
kFsFx(k64f) kLc900xInfo_AnalogGainFactor(kLc900xInfo info, k32u index);
kFsFx(k32u) kLc900xInfo_DigitalGainStep(kLc900xInfo info, k64f gain);
kFsFx(k64f) kLc900xInfo_DigitalGainFactor(kLc900xInfo info, k32u index);

kFsFx(k64f) kLc900xInfo_RowTime(kLc200xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc900xInfo_ReadoutTime(kLc900xInfo info, const kCameraInfoParams* params);
kFsFx(k64f) kLc900xInfo_FrameOverheadTime(kLc200xInfo info, const kCameraInfoParams* params);

#endif
