/** 
 * @file    kCameraTracking.h
 * @brief   Declares the kCameraTracking type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CAMERA_TRACKING_H
#define K_FIRESYNC_CAMERA_TRACKING_H

#include <kFireSync/Client/Camera/Extensions/kCameraExt.h>

/** @relates kCameraTracking @{ */
#define kCAMERA_TRACKING_STATE_SEARCH            (0)         ///< Index of search state.
#define kCAMERA_TRACKING_STATE_TRACKING_0        (1)         ///< Index of first tracking state.
/** @} */

/**
 * @class   kCameraTracking
 * @extends kCameraExt
 * @ingroup kFireSync-Client-Camera-Extensions
 * @brief   Represents parameters for a camera tracking extension. 
 * 
 * This extension enables the camera to automatically track the position of a laser line, dynamically 
 * adjusting the y-origin of the camera window in response to changes in the laser position. For acquisition 
 * systems with targets that remain relatively flat within the scope of a single camera frame, this can enable 
 * the use of a smaller ROI (yielding higher frame rates), without sacrificing overall measurement range across 
 * successive camera frames. 
 * 
 * The tracking algorithm works by dynamically switching between <em>search</em> and <em>tracking</em>
 * camera states. By convention, camera state 0 is considered the search state and camera state 1 is 
 * the first tracking state. These two states are required at a minimum; if additional states are defined, 
 * they are treated as extra tracking states. 
 * 
 * In a typical tracking setup, the search state (camera state 0) is configured with a relatively large 
 * camera height value, so that the entirety of the measurement range can be observed in a single frame. 
 * The tracking state(s), in turn, are configured with smaller height values, enabling higher frame rates. 
 * Under ideal operation, the laser signal will be detected in a search frame and the tracking algorithm
 * will then switch to tracking mode, dynamically adjusting the camera ROI y-position to keep the laser 
 * signal in the frame. The algorithm will remain in tracking mode indefinitely, only dropping back 
 * to search mode when the laser signal is lost. 
 * 
 * The tracking algorithm proceeds from the search state to the first tracking state when the 
 * number of spots detected in the image is greater than or equal to the configured <em>search threshold</em>. 
 * The tracking algorithm will revert from tracking to search mode when the number of detected spots is 
 * less than the search threshold. If multiple tracking states are defined, the algorithm will always execute 
 * the complete set of tracking frames before switching modes or adjusting the ROI y-position. The 
 * determination about whether to switch modes is made after the final tracking frame is processed. 
 * 
 * In current implementations, the window tracking algorithm is implemented through a combination of PL 
 * and CPU driver support. Accordingly, this extension requires a PL build with a column-based CG algorithm 
 * (e.g., Sobelv_Accgdw) and for the PL algorithm to be enabled. The PL algorithm performs bulk image
 * analysis, while the CPU driver makes high level decisions about state changes.
 * 
 * Due to the time required for PL image processing and CPU decision-making, and the requirement to configure 
 * subsequent frames in advance of imager exposure, there is a 3-frame delay between when images are acquired 
 * and when corresponding tracking state changes are enacted. 
 * 
 * Due to the use of CPU drivers, there is a non-negligible CPU run-time cost when this feature is used at 
 * high frame rates. At the time of this writing, current M1, M2, and M3 implementations have been established 
 * to work reliably at up to 5kHz. Higher frame rates may be possible (e.g., some tests have been successful 
 * at 7-8 kHz), but are not recommended in general. If run too fast, the consequence is likely to be dropped 
 * interrupts, with corrupted and/or out-of-sequence camera images as a secondary consequence.
 */
//typedef kCameraExt kCameraTracking;            --forward-declared in kFsDef.x.h

/** 
 * Sets the tracking search threshold. 
 * 
 * The tracking algorithm will proceed from search mode to tracking mode when the number of 
 * detected spots is greater than or equal to the search threshold. The tracking algorithm will revert
 * from tracking mode to search mode when the number of detected spots is less than the search 
 * threshold. There is no hysteresis threshold to debounce state changes. 
 * 
 * The search threshold is specified in terms of the number of spots that are expected to be 
 * observed in the search state. The expected number of spots may differ between search and tracking 
 * states due to x-subsampling parameter changes between states. The tracking algorithm automatically 
 * adjusts the search threshold to compensate for these changes.
 * 
 * @public              @memberof kCameraTracking
 * @param   extension   Extension object. 
 * @param   threshold   Search threshold. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraTracking_SetSearchThreshold(kCameraTracking extension, k32u threshold); 

/** 
 * Gets the tracking search threshold. 
 *
 * @public              @memberof kCameraTracking
 * @param   extension   Extension object. 
 * @return              Search threshold.  
 * @see                 kCameraTracking_SetSearchThreshold
 */
kFsFx(k32u) kCameraTracking_SearchThreshold(kCameraTracking extension); 

#include <kFireSync/Client/Camera/Extensions/kCameraTracking.x.h>

#endif
