/** 
 * @file    kCameraSteps.h
 * @brief   Declares the kCameraSteps type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CAMERA_STEPS_H
#define K_FIRESYNC_CAMERA_STEPS_H

#include <kFireSync/Client/Camera/Extensions/kCameraExt.h>

/**
 * @class   kCameraSteps
 * @extends kCameraExt
 * @ingroup kFireSync-Client-Camera-Extensions
 * @brief   Represents parameters for a camera step extension. 
 */
//typedef kCameraExt kCameraSteps;            --forward-declared in kFsDef.x.h

/** 
 * Sets the number of steps (state/count pairs) defined by this extension. 
 *
 * @public              @memberof kCameraSteps
 * @param   extension   Extension object. 
 * @param   count       Step count. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraSteps_SetCount(kCameraSteps extension, kSize count); 

/** 
 * Gets the number of steps (state/count pairs) defined by this extension. 
 *
 * @public              @memberof kCameraSteps
 * @param   extension   Sequence object. 
 * @return              Step count. 
 */
kFsFx(kSize) kCameraSteps_Count(kCameraSteps extension); 

/** 
 * Reports constraint and validity information for the Count setting. 
 *
 * @public              @memberof kCameraSteps
 * @param   extension   Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraSteps_CountInfo(kCameraSteps extension, kInfoSize* info); 

/** 
 * Sets the camera state id (index) associated with the specified step index. 
 *
 * @public              @memberof kCameraSteps
 * @param   extension   Extension object. 
 * @param   stepIndex   Step index. 
 * @param   stateIndex  Camera state id/index. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraSteps_SetState(kCameraSteps extension, kSize stepIndex, kSize stateIndex); 

/** 
 * Gets the camera state id (index) associated with the specified step index. 
 *
 * @public              @memberof kCameraSteps
 * @param   extension   Extension object. 
 * @param   stepIndex   Step index. 
 * @return              Camera state id/index.
 */
kFsFx(kSize) kCameraSteps_State(kCameraSteps extension, kSize stepIndex); 

/** 
 * Sets the repetition count associated with the specified step index. 
 *
 * @public              @memberof kCameraSteps
 * @param   extension   Sequence object. 
 * @param   stepIndex   Step index. 
 * @param   count       Repetition count. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraSteps_SetRepetitionCount(kCameraSteps extension, kSize stepIndex, kSize count); 

/** 
 * Gets the repetition count associated with the specified step index. 
 *
 * @public              @memberof kCameraSteps
 * @param   extension   Extension object. 
 * @param   stepIndex   Step index. 
 * @return              Repetition count. 
 */
kFsFx(kSize) kCameraSteps_RepetitionCount(kCameraSteps extension, kSize stepIndex); 

/**
* Sets the steps style.
*
* @public              @memberof kCameraSteps
* @param   extension   Sequence object.
* @param   style       Steps style.
* @return              Operation status.
*/
kFsFx(kStatus) kCameraSteps_SetStyle(kCameraSteps extension, kCameraStepsStyle style);

/**
* Gets the steps style.
*
* @public              @memberof kCameraSteps
* @param   extension   Extension object.
* @return              Steps style.
*/
kFsFx(kCameraStepsStyle) kCameraSteps_Style(kCameraSteps extension);

/**
* Reports constraint and validity information for the Steps style setting.
*
* @public              @memberof kCameraSteps
* @param   extension   Extension object.
* @param   info        Receives the info structure.
* @return              Operation status.
*/
kFsFx(kStatus) kCameraSteps_StyleInfo(kCameraSteps extension, kInfoBits* info);

#include <kFireSync/Client/Camera/Extensions/kCameraSteps.x.h>

#endif
