/** 
 * @file    kCameraDynExp.h
 * @brief   Declares the kCameraDynExp type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_CAMERA_DYN_EXP_H
#define K_FIRESYNC_CAMERA_DYN_EXP_H

#include <kFireSync/Client/Camera/Extensions/kCameraExt.h>

/**
 * @class   kCameraDynExp
 * @extends kCameraExt
 * @ingroup kFireSync-Client-Camera-Extensions
 * @brief   Represents parameters for a camera dynamic exposure sequence. 
 */
//typedef kCameraExt kCameraDynExp;            --forward-declared in kFsDef.x.h

/** 
 * Sets the minimum exposure parameter, in nanoseconds.
 *
 * @public              @memberof kCameraDynExp
 * @param   sequence    Extension object. 
 * @param   exposure    Minimum exposure value, in nanoseconds.  
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraDynExp_SetMinExposure(kCameraDynExp sequence, k64u exposure, kAdjust adjustment = kADJUST_NONE); 

/** 
 * Gets the minimum exposure parameter, in nanoseconds.
 *
 * @public              @memberof kCameraDynExp
 * @param   sequence    Extension object. 
 * @return              Minimum exposure parameter, in nanoseconds.  
 */
kFsFx(k64u) kCameraDynExp_MinExposure(kCameraDynExp sequence); 

/** 
 * Reports constraint and validity information for the MinExposure setting. 
 *
 * @public              @memberof kCameraDynExp
 * @param   sequence    Extension object. 
 * @param   info        Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraDynExp_MinExposureInfo(kCameraDynExp sequence, kInfo64u* info); 

/** 
 * Sets the minimum spots parameter.
 *
 * @public              @memberof kCameraDynExp
 * @param   sequence    Extension object. 
 * @param   minSpots    Minimum spots parameter.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraDynExp_SetMinSpots(kCameraDynExp sequence, k32u minSpots); 

/** 
 * Gets the minimum spots parameter.
 *
 * @public              @memberof kCameraDynExp
 * @param   sequence    Extension object. 
 * @return              Minimum spots parameter.  
 */
kFsFx(k32u) kCameraDynExp_MinSpots(kCameraDynExp sequence); 

/** 
 * Gets the number of entries in the adjustment table. 
 * 
 * The number of entries is not configurable. 
 *
 * @public              @memberof kCameraDynExp
 * @param   sequence    Extension object. 
 * @return              Count of entries in the adjustment table. 
 */
kFsFx(kSize) kCameraDynExp_EntryCount(kCameraDynExp sequence); 

/** 
 * Sets the spot threshold associated with the specified adjustment table row index. 
 *
 * @public              @memberof kCameraDynExp
 * @param   sequence    Extension object. 
 * @param   entryIndex  Adjustment table row index. 
 * @param   threshold   Spot threshold. 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraDynExp_SetSpotThreshold(kCameraDynExp sequence, kSize entryIndex, k32u threshold); 

/** 
 * Gets the spot threshold associated with the specified adjustment table row index. 
 *
 * @public              @memberof kCameraDynExp
 * @param   sequence    Extension object. 
 * @param   entryIndex  Adjustment table row index. 
 * @return              Spot threshold. 
 */
kFsFx(k32u) kCameraDynExp_SpotThreshold(kCameraDynExp sequence, kSize entryIndex); 

/** 
 * Sets the adjustment percentage associated with the specified adjustment table row index. 
 *
 * @public              @memberof kCameraDynExp
 * @param   sequence    Extension object. 
 * @param   entryIndex  Adjustment table row index. 
 * @param   adjust      Signed adjustment percentage value (-100..100). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kCameraDynExp_SetAdjust(kCameraDynExp sequence, kSize entryIndex, k32s adjust); 

/** 
 * Gets the adjustment percentage associated with the specified adjustment table row index. 
 *
 * @public              @memberof kCameraDynExp
 * @param   sequence    Extension object. 
 * @param   entryIndex  Adjustment table row index. 
 * @return              Signed adjustment percentage value (-100..100). 
 */
kFsFx(k32s) kCameraDynExp_Adjust(kCameraDynExp sequence, kSize entryIndex); 

#include <kFireSync/Client/Camera/Extensions/kCameraDynExp.x.h>

#endif
