/** 
 * @file    kBinCSum4Info.h
 * @brief   Declares the kBinCSum4Info type. 
 *
 * @internal
 * Copyright (C) 2016-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_BINCSUM4_INFO_H
#define K_FIRESYNC_BINCSUM4_INFO_H

#include <kFireSync/Client/Camera/Algorithms/kAlgInfo.h>

/**
 * @class   kBinCSum4Info
 * @extends kAlgInfo
 * @ingroup kFireSync-Client-Camera-Algorithms
 * @brief   Represents configuration for the BIN_CSUM4 (Binarize + Column-Sum, version 4) algorithm.
 */
//typedef kAlgInfo kBinCSum4Info;            --forward-declared in kFsDef.x.h

/** 
 * Constructs a kBinCSum4Info object.
 *
 * @public                  @memberof kBinCSum4Info
 * @param   info            Destination for the constructed object handle. 
 * @param   capabilities    Represents camera features.
 * @param   plConfig        PL configuration id. 
 * @param   plVersion       PL configuration version.
 * @param   allocator       Memory allocator (or kNULL for default). 
 * @return                  Operation status. 
 */
kFsFx(kStatus) kBinCSum4Info_Construct(kBinCSum4Info* info, kCameraCapability capabilities, k32u plConfig, kVersion plVersion, kAlloc allocator); 

/** @name BinThreshold */
//@{

/** 
 * Sets the binarize threshold. 
 *
 * @public              @memberof kBinCSum4Info
 * @param   info        Algorithm info object. 
 * @param   threshold   Binarize threshold. 
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Info_SetBinThreshold(kBinCSum4Info info, k32u threshold, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the binarize threshold. 
 *
 * @public              @memberof kBinCSum4Info
 * @param   info        Algorithm info object. 
 * @return              Binarize threshold. 
 */
kFsFx(k32u) kBinCSum4Info_BinThreshold(kBinCSum4Info info);

/** 
 * Reports constraint and validity information for the BinThreshold setting. 
 *
 * @public              @memberof kBinCSum4Info
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Info_BinThresholdInfo(kAccgd2kInfo info, kInfo32u* info32u);

//@}
/** @name CSumThreshold */
//@{

/** 
 * Sets the column-sum threshold. 
 *
 * @public              @memberof kBinCSum4Info
 * @param   info        Algorithm info object. 
 * @param   threshold   Column-sum threshold. 
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Info_SetCSumThreshold(kBinCSum4Info info, k32u threshold, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the column-sum threshold. 
 *
 * @public              @memberof kBinCSum4Info
 * @param   info        Algorithm info object. 
 * @return              Column-sum threshold. 
 */
kFsFx(k32u) kBinCSum4Info_CSumThreshold(kBinCSum4Info info);

/** 
 * Reports constraint and validity information for the CSumThreshold setting. 
 *
 * @public              @memberof kBinCSum4Info
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Info_CSumThresholdInfo(kAccgd2kInfo info, kInfo32u* info32u);

//@}
/** @name BackgroundWidth */
//@{

/** 
 * Sets the number of left-most columns used to calculate background noise.  
 * 
 * A value of 0 can be specified to disable background noise suppression.
 *
 * @public              @memberof kBinCSum4Info
 * @param   info        Algorithm info object. 
 * @param   width       Column count, prior to any subsampling.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kBinCSum4Info_SetBackgroundWidth(kBinCSum4Info info, kSize width, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the number of left-most columns used to calculate background noise.  
 *
 * @public              @memberof kBinCSum4Info
 * @param   info        Algorithm info object. 
 * @return              Column count, prior to any subsampling.
 */
kFsFx(kSize) kBinCSum4Info_BackgroundWidth(kBinCSum4Info info);

/** 
 * Reports constraint and validity information for the BackgroundColumnCoun setting. 
 *
 * @public                       @memberof kBinCSum4Info
 * @param   info                 Algorithm info object.  
 * @param   backgroundWidthInfo  Receives the info structure.  
 * @return                       Operation status. 
 */
kFsFx(kStatus) kBinCSum4Info_BackgroundWidthInfo(kAccgd2kInfo info, kInfoSize* backgroundWidthInfo);

//@}

#include <kFireSync/Client/Camera/Algorithms/kBinCSum4Info.x.h>

#endif
