/** 
 * @file    kAcgInfo.h
 * @brief   Declares the kAcgInfo type. 
 *
 * @internal
 * Copyright (C) 2013-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_ACG_INFO_H
#define K_FIRESYNC_ACG_INFO_H

#include <kFireSync/Client/Camera/Algorithms/kAlgInfo.h>

/**
 * @class   kAcgInfo
 * @extends kAlgInfo
 * @ingroup kFireSync-Client-Camera-Algorithms
 * @brief   Abstract base class for laser-line spot-detection camera algorithms. 
 */
//typedef kAlgInfo kAcgInfo;            --forward-declared in kFsDef.x.h

/** 
 * Reports whether the algorithm is row-based. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object. 
 * @return              Operation status. 
 */
kFsFx(kBool) kAcgInfo_IsRowBased(kAcgInfo info);

/**
* Reports the spot metadata scale factor.
* 
* The spot metadata scale factor represents the amount by which spot metadata fields (i.e., width, sum)
* are divided down. This division enables larger results to be communicated within a fixed number of bits 
* in the spot data structure. 
*
* @public              @memberof kAcgInfo
* @param   info        Algorithm info object.
* @return              Operation status.
*/
kFsFx(kSize) kAcgInfo_CgScale(kAcgInfo info);

/** @name RangesEnabled */
//@{

/** 
 * Specifies whether range lookup is enabled.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   enabled     Should range lookup be enabled?
 * @return              Operation status. 
 */
kFsFx(kStatus) kAcgInfo_EnableRanges(kAcgInfo info, kBool enabled);

/** 
 * Reports whether range lookup is enabled.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @return              kTRUE if ranges are enabled.
 */
kFsFx(kBool) kAcgInfo_RangesEnabled(kAcgInfo info);

/** 
 * Reports constraint and validity information for the EnableRanges setting. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   infoBool    Receives the info structure.  
 * @return              Operation status. 
 */ 
kFsFx(kStatus) kAcgInfo_RangesEnabledInfo(kAcgInfo info, kInfoBool* infoBool);

//@}
/** @name RangeSlicesEnabled */
//@{

/** 
 * Specifies whether range slice (index) information is enabled.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   enabled     Should range slice information be enabled?
 * @return              Operation status. 
 */
kFsFx(kStatus) kAcgInfo_EnableRangeSlices(kAcgInfo info, kBool enabled);

/** 
 * Reports whether range slice (index) information is enabled.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @return              kTRUE if range slices are enabled.
 */
kFsFx(kBool) kAcgInfo_RangeSlicesEnabled(kAcgInfo info);

/** 
 * Reports constraint and validity information for the EnableRangeSlices setting. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   infoBool    Receives the info structure.  
 * @return              Operation status. 
 */ 
kFsFx(kStatus) kAcgInfo_RangeSlicesEnabledInfo(kAcgInfo info, kInfoBool* infoBool);

//@}
/** @name ResampleEnabled */
//@{

/** 
 * Specifies whether range resampling is enabled.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   enabled     Should range resampling be enabled?
 * @return              Operation status. 
 */
kFsFx(kStatus) kAcgInfo_EnableResample(kAcgInfo info, kBool enabled);

/** 
 * Reports whether range resampling is enabled.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @return              kTRUE if range resampling is enabled. 
 */
kFsFx(kBool) kAcgInfo_ResampleEnabled(kAcgInfo info);

/** 
 * Reports constraint and validity information for the EnableResample setting. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   infoBool    Receives the info structure.  
 * @return              Operation status. 
 */ 
kFsFx(kStatus) kAcgInfo_ResampleEnabledInfo(kAcgInfo info, kInfoBool* infoBool);

//@}
/** @name ResamplePreferHigh */
//@{

/** 
 * Specifies whether higher or lower ranges are selected when there are multiple choices.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   preferHigh  Higher range values should be selected.
 * @return              Operation status. 
 */
kFsFx(kStatus) kAcgInfo_SetResamplePreferHigh(kAcgInfo info, kBool preferHigh);

/** 
 * Reports whether higher or lower ranges are selected when there are multiple choices.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @return              kTRUE if higher values are selected. 
 */
kFsFx(kBool) kAcgInfo_ResamplePreferHigh(kAcgInfo info);

/** 
 * Reports constraint and validity information for the ResamplePreferHigh setting. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   infoBool    Receives the info structure.  
 * @return              Operation status. 
 */ 
kFsFx(kStatus) kAcgInfo_ResamplePreferHighInfo(kAcgInfo info, kInfoBool* infoBool);
 
//@}
/** @name ResampleBegin */
//@{

/** 
 * Specifies the initial (left-most) position for x-resampling. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   begin       Starting X-coordinate for resampling.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kAcgInfo_SetResampleBegin(kAcgInfo info, k32s begin, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the initial (left-most) position for x-resampling. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @return              Starting X-coordinate for resampling. 
 */
kFsFx(k32s) kAcgInfo_ResampleBegin(kAcgInfo info);

/** 
 * Reports constraint and validity information for the ResampleBegin setting. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   info32s     Receives the info structure.  
 * @return              Operation status. 
 */ 
kFsFx(kStatus) kAcgInfo_ResampleBeginInfo(kAcgInfo info, kInfo32s* info32s);
 
//@}
/** @name ResampleCount */
//@{

/** 
 * Specifies the total number of x-resampling points. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   count       Total number of x-resampling points.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kAcgInfo_SetResampleCount(kAcgInfo info, k32u count, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the total number of x-resampling points. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @return              Total number of x-resampling points.
 */
kFsFx(k32u) kAcgInfo_ResampleCount(kAcgInfo info);

/** 
 * Reports constraint and validity information for the ResampleCount setting. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */ 
kFsFx(kStatus) kAcgInfo_ResampleCountInfo(kAcgInfo info, kInfo32u* info32u);
 
//@}
/** @name ResampleStep */
//@{

/** 
 * Specifies the distance in the x-axis between resampled points. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   step        Distance bewteen adjacent resampled points.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kAcgInfo_SetResampleStep(kAcgInfo info, k32u step, kAdjust adjustment = kADJUST_NONE);

 /** 
 * Reports the distance in the x-axis between resampled points. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @return              Distance bewteen adjacent resampled points. 
 */
kFsFx(k32u) kAcgInfo_ResampleStep(kAcgInfo info);

/** 
 * Reports constraint and validity information for the ResampleStep setting. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */ 
kFsFx(kStatus) kAcgInfo_ResampleStepInfo(kAcgInfo info, kInfo32u* info32u);
 
//@}
/** @name ResampleStepThreshold */
//@{

/** 
 * Specifies the maximum distance over which to interpolate between input points when resampling.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   threshold   Maximum distance over which to interpolate.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kAcgInfo_SetResampleStepThreshold(kAcgInfo info, k32u threshold, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the maximum distance over which to interpolate between input points when resampling.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @return              Maximum distance over which to interpolate. 
 */
kFsFx(k32u) kAcgInfo_ResampleStepThreshold(kAcgInfo info);

/** 
 * Reports constraint and validity information for the ResampleStepThreshold setting. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   info32u     Receives the info structure.  
 * @return              Operation status. 
 */ 
kFsFx(kStatus) kAcgInfo_ResampleStepThresholdInfo(kAcgInfo info, kInfo32u* info32u);

//@}
/** @name ResampleZMin */
//@{

/** 
 * Specifies the minimum z threshold for resampled points.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   min         Minimum z-threshold (inclusive) for resampled points.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kAcgInfo_SetResampleZMin(kAcgInfo info, k32s min, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the minimum z threshold for resampled points.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.      
 * @return              Minimum z-threshold (inclusive) for resampled points.
 */
kFsFx(k32s) kAcgInfo_ResampleZMin(kAcgInfo info);

/** 
 * Reports constraint and validity information for the ResampleZMin setting. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   info32s     Receives the info structure.  
 * @return              Operation status. 
 */ 
kFsFx(kStatus) kAcgInfo_ResampleZMinInfo(kAcgInfo info, kInfo32s* info32s);

//@}
/** @name ResampleZMax */
//@{

/** 
 * Specifies the maximum z threshold for resampled points.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   max         Maximum z-threshold (inclusive) for resampled points.
 * @param   adjustment  Automatic adjustment style (optional).
 * @return              Operation status. 
 */
kFsFx(kStatus) kAcgInfo_SetResampleZMax(kAcgInfo info, k32s max, kAdjust adjustment = kADJUST_NONE);

/** 
 * Reports the maximum z threshold for resampled points.
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.      
 * @return              Maximum z-threshold (inclusive) for resampled points.
 */
kFsFx(k32s) kAcgInfo_ResampleZMax(kAcgInfo info);

/** 
 * Reports constraint and validity information for the ResampleZMax setting. 
 *
 * @public              @memberof kAcgInfo
 * @param   info        Algorithm info object.  
 * @param   info32s     Receives the info structure.  
 * @return              Operation status. 
 */ 
kFsFx(kStatus) kAcgInfo_ResampleZMaxInfo(kAcgInfo info, kInfo32s* info32s);
 
//@}

#include <kFireSync/Client/Camera/Algorithms/kAcgInfo.x.h>

#endif
