/** 
 * @file    kMemoryDomainBlockInfo.h
 * @brief   Declares the kMemoryDomainBlockInfo class. 
 *
 * @internal
 * Copyright (C) 2020-2022 by LMI Technologies Inc.  All rights reserved.
 */
#ifndef K_FIRESYNC_MEMORY_DOMAIN_BLOCK_INFO_H
#define K_FIRESYNC_MEMORY_DOMAIN_BLOCK_INFO_H

#include <kFireSync/kNodeDef.h>

/**
 * @struct  kMemoryDomainBlockOutput
 * @extends kValue
 * @ingroup kFireSync-Blocks
 * @brief   Represents a kMemoryDomainBlock output memory type. 
 */
typedef k32s kMemoryDomainBlockOutput; 

/** @relates kMemoryDomainBlockOutput @{ */
#define kMEMORY_DOMAIN_BLOCK_OUTPUT_HOST               (0x01)       ///< Any host-accessible memory.
#define kMEMORY_DOMAIN_BLOCK_OUTPUT_CUDA               (0x02)       ///< Any cuda device-accessible memory. 
#define kMEMORY_DOMAIN_BLOCK_OUTPUT_CUDA_DEVICE        (0x04)       ///< Cuda device memory.
#define kMEMORY_DOMAIN_BLOCK_OUTPUT_CUDA_MANAGED       (0x08)       ///< Cuda managed memory.
#define kMEMORY_DOMAIN_BLOCK_OUTPUT_CUDA_PINNED        (0x10)       ///< Cuda pinned memory.
#define kMEMORY_DOMAIN_BLOCK_OUTPUT_OPTIONS            (0x1F)       ///< Bitset containing all valid options.
/** @} */

#include <kFireSync/Blocks/kMemoryDomainBlockInfo.x.h>

/**
 * @class   kMemoryDomainBlockInfo
 * @extends kObject
 * @ingroup kFireSync-Blocks
 * @brief   Represents configuration for kMemoryDomainBlock. 
 */
//typedef kObject kMemoryDomainBlockInfo;        --forward-declared in kFsDef.x.h

/** 
 * Constructs a kMemoryDomainBlockInfo object. 
 *
 * @public              @memberof kMemoryDomainBlockInfo
 * @param   info        Receives constructed info object. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
 kFsFx(kStatus) kMemoryDomainBlockInfo_Construct(kMemoryDomainBlockInfo* info, kAlloc allocator); 

/** 
 * Constructs a kMemoryDomainBlockInfo object and loads its content from an XML object. 
 *
 * @public              @memberof kMemoryDomainBlockInfo
 * @param   info        Receives constructed info object.  
 * @param   xml         Source from which parameters are loaded. 
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kMemoryDomainBlockInfo_Load(kMemoryDomainBlockInfo* info, kXml xml, kAlloc allocator); 

/** 
 * Constructs a kXml object and uses it to store the content of this kMemoryDomainBlockInfo instance.
 *
 * @public              @memberof kMemoryDomainBlockInfo
 * @param   info        Info object.  
 * @param   xml         Receives constructed kXml object.
 * @param   allocator   Memory allocator (or kNULL for default). 
 * @return              Operation status. 
 */
kFsFx(kStatus) kMemoryDomainBlockInfo_Store(kMemoryDomainBlockInfo info, kXml* xml, kAlloc allocator); 

/** 
 * Reads kMemoryDomainBlockInfo state from a kXml object. 
 *
 * @public              @memberof kMemoryDomainBlockInfo
 * @param   info        Info object.  
 * @param   xml         XML object. 
 * @param   item        XML item.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kMemoryDomainBlockInfo_Read(kMemoryDomainBlockInfo info, kXml xml, kXmlItem item); 

/** 
 * Writes kMemoryDomainBlockInfo state to a kXml object. 
 *
 * @public              @memberof kMemoryDomainBlockInfo
 * @param   info        Info object.  
 * @param   xml         XML object. 
 * @param   item        XML item.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kMemoryDomainBlockInfo_Write(kMemoryDomainBlockInfo info, kXml xml, kXmlItem item); 

/** 
 * Checks and automatically corrects all state variables. 
 *
 * @public              @memberof kMemoryDomainBlockInfo
 * @param   info        Info object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kMemoryDomainBlockInfo_Validate(kMemoryDomainBlockInfo info); 

/** 
 * Resets all state variables to default values.
 *
 * @public              @memberof kMemoryDomainBlockInfo
 * @param   info        Info object.  
 * @return              Operation status. 
 */
kFsFx(kStatus) kMemoryDomainBlockInfo_Clear(kMemoryDomainBlockInfo info); 

/** 
 * Enables forced cloning, which causes the block to clone even when incoming memory may already be suitable. 
 *
 * @public              @memberof kMemoryDomainBlockInfo
 * @param   info        Info object.  
 * @param   enabled     kTRUE to enable.
 * @return              Operation status. 
 */
kInlineFx(kStatus) kMemoryDomainBlockInfo_EnableForceClone(kMemoryDomainBlockInfo info, kBool enabled)
{
    kObj(kMemoryDomainBlockInfo, info); 

    obj->forceCloneEnabled = enabled;

    return kOK; 
}

/** 
 * Reports whether forced cloning is enabled. 
 *
 * @public              @memberof kMemoryDomainBlockInfo
 * @param   info        Info object.  
 * @return              kTRUE if enabled.
 */
kInlineFx(kBool) kMemoryDomainBlockInfo_ForceCloneEnabled(kMemoryDomainBlockInfo info)
{
    kObj(kMemoryDomainBlockInfo, info); 

    return obj->forceCloneEnabled; 
}

/** 
 * Sets the desired output memory type.
 * 
 * @public              @memberof kMemoryDomainBlockInfo
 * @param   info        Info object.  
 * @param   outputType  Output memory type.
 * @return              Operation status. 
 */
kInlineFx(kStatus) kMemoryDomainBlockInfo_SetOutputMemory(kMemoryDomainBlockInfo info, kMemoryDomainBlockOutput outputType)
{
    kObj(kMemoryDomainBlockInfo, info); 

    obj->outputMemory = outputType;

    return kOK; 
}

/** 
 * Reports the selected output memory type. 
 *
 * @public              @memberof kMemoryDomainBlockInfo
 * @param   info        Info object.  
 * @return              Memory type.
 */
kInlineFx(kMemoryDomainBlockOutput) kMemoryDomainBlockInfo_OutputMemory(kMemoryDomainBlockInfo info)
{
    kObj(kMemoryDomainBlockInfo, info); 

    return obj->outputMemory;
}

/** Gets the input port id. */
kInlineFx(k16u) kMemoryDomainBlockInfo_InputPortId() { return 0;}

/** Gets the output port id. */
kInlineFx(k16u) kMemoryDomainBlockInfo_OutputPortId() { return 1; }

#endif
