import os
from os import path
from glob import glob
from enum import Enum
import argparse
import re
import tarfile # import TarFile, TarInfo
from xml.etree.ElementTree import Element, ElementTree, parse, SubElement, fromstring as str2etree, tostring as etree2str
from io import BytesIO
import time

GDK_MANIFEST_XML = 'gdk_manifest.xml'

PROJECT_XML_TEMPLATE = '''
  <Project name="VETools/AddOnTools/{ToolName}.xml">
    <Classification options="Standard|FAE|Sales|Library">{Classification}</Classification>
  </Project>
'''
VAR_TEMPLATE = '#[{}]'

class Platforms(Enum):
    ARM7        = ('.kext', '', '')
    LINUX_ARM64 = ('.so'  , '/app_main/lib/linux_arm64/', 'lib')
    WIN32       = ('.dll' , '/bin/win32', '')
    WIN64       = ('.dll' , '/bin/win64', '')
    WIN32D      = ('.dll' , '/bin/win32d', '')
    WIN64D      = ('.dll' , '/bin/win64d', '')
    
    def isWin(self):
        return self is Platforms.WIN32 or \
               self is Platforms.WIN64
    def ext(self):
        return self.value[0]
    def path(self):
        return self.value[1]
    def prefix(self):
        return self.value[2]
    def platName(self):
        return self.name.lower()
class GdkManifest:
    def __init__(self, gocatorVersion):
        self.root = Element("GdkAppPackage")
        xmlGocVersion = SubElement(self.root, 'GocatorVersion')
        xmlGocVersion.text = gocatorVersion
    @classmethod
    def fromString(cls, text):
        c = cls('')
        c.root = str2etree(text)
        return c
    def addFile(self, gdkAppName, platform, type, source, destination="", version = None):
        xmlGdkApp = None
        for xmlGdkApp_ in self.root.iter("GdkApp"):
            if xmlGdkApp_.get('name') == gdkAppName:
                xmlGdkApp = xmlGdkApp_
                break
        if not xmlGdkApp:
            xmlGdkApp = SubElement(self.root, 'GdkApp')
            xmlGdkApp.set('name', gdkAppName)

        xmlPlatform = None
        for xmlPlatform_ in self.root.iter("Platform"):
            if xmlPlatform_.get('name') == platform.platName():
                xmlPlatform = xmlPlatform_
                break
        if not xmlPlatform:
            xmlPlatform = SubElement(xmlGdkApp, 'Platform')
            xmlPlatform.set('name', platform.platName())
        
        xmlFile = SubElement(xmlPlatform, "File")
        xmlFile.set('type', type)
        if version:
            xmlFile.set('version', version)
        xmlSource = SubElement(xmlFile, "Source")
        xmlSource.text = source
        xmlDestination = SubElement(xmlFile, "Destination")
        xmlDestination.text = path.join(destination, path.basename(source))
    def getFiles(self, platform = None):
        allFiles = []
        for xmlGdkApp in self.root.iter("GdkApp"):
            for xmlPlatform in xmlGdkApp.iter("Platform"):
                if platform is None or platform.platName() == xmlPlatform.get('name'):
                    for xmlFile in xmlPlatform.iter("File"):
                        allFiles.append(path.basename(xmlFile.find('Source').text))
        return allFiles

    def save(self, fileName, pretty = False):
        if pretty:
            with open(fileName, 'w') as oFile:
                oFile.write(self.tostring())
        else:
            ElementTree(self.root).write(fileName)
    def tostring(self):
        import xml.dom.minidom as minidom
        return minidom.parseString(etree2str(self.root, 'utf-8')).toprettyxml()
    def toTar(self, fileName):
        manifestData = self.tostring().encode('utf8')
        manifestTarinfo = tarfile.TarInfo(fileName)
        manifestTarinfo.size = len(manifestData)
        return manifestTarinfo, BytesIO(manifestData)

class Classification(Enum):
    Standard    = { 'index':0                  }
    FAE         = { 'index':1,  'beta':True    }
    Sales       = { 'index':2,  'beta':True    }
    Library     = { 'index':3                  }

    def isTool(self):
        return self is not Classification.Library
    def isBeta(self):
        return self.value.get('beta', False)
    def label(self):
        if self.isBeta():
            return "Beta_" + self.name
        else:
            return self.name
    @classmethod
    def get(cls, name, case_sensitive=True):
        if case_sensitive:
            return cls[name]
        else:
            for c in Classification:
                if c.name.lower() == name.lower():
                    return c
class XmlWrapper:
    def __init__(self, name, log=print, workdir=None):
        self.path = name
        if not self.path.endswith('.xml'):
            self.path = self.path + '.xml'
        if workdir is not None:
            self.path = path.join(workdir, self.path)
        self.xmlContent = parse(self.path)
        self.log = log
    def save(self):
        self.xmlContent.write(self.path)
class ProjectXml(XmlWrapper):
    def replaceVariable(self, oldVarName, newVarName, newVarValue):
        self.removeVariable(oldVarName)
        self.addVariable(newVarName, newVarValue)
        self.replaceRecursive({VAR_TEMPLATE.format(oldVarName):VAR_TEMPLATE.format(newVarName)})
    def removeVariable(self, varName):
        xmlVariables = self.xmlContent.find('Variables')
        for xmlVar in xmlVariables:            
            if xmlVar.get('name') == varName:
                xmlVariables.remove(xmlVar)
                break
    def replaceRecursive(self, textMap, parent=None):
        if parent is None:
            parent = self.xmlContent.getroot()
        for child in parent:
            if child.text:
                for oldText in textMap.keys():
                    if oldText in child.text:
                        child.text = child.text.replace(oldText, textMap[oldText])
            self.replaceRecursive(textMap, child)
            
    def addVariable(self, varName, value):
        xmlVar = Element('Variable', name=varName)
        xmlVar.text = value

        xmlVariables = self.xmlContent.find('Variables')
        xmlVariables.append(xmlVar)
    def removeConfiguration(self, platform, name=None):
        xmlConfigs = self.xmlContent.find('Configurations')
        for xmlConfig in xmlConfigs.getchildren():
            if xmlConfig.findtext('Platform') == platform:
                if name is None or name == xmlConfig.findtext('Name'):
                    xmlConfigs.remove(xmlConfig)
    def addConfiguration(self, string=None):
        if string is not None:
            xmlConfig = str2etree(string)
            xmlConfig.tail = '\n    '
            self.xmlContent.find('Configurations').append(xmlConfig)
        else:
            raise Exception('Other methods not implemented')
class SolutionXml(XmlWrapper):

    def getProjects(self):
        tools = []
        for xmlProj in self.xmlContent.iter('Project'):            
            if Classification[xmlProj.findtext('Classification')].isTool():
                tools.append(path.splitext(path.basename(xmlProj.get('name')))[0])
        return tools
    def addProject(self, toolName, classification, dependedncies):
        xmlProject = str2etree(PROJECT_XML_TEMPLATE.format(
            ToolName = toolName,
            Classification = classification.name
        ))
        xmlProject.tail = '\n  '

        for d in dependedncies:
            xmlDepend = Element("Dependency", {'name':d})
            xmlProject.append(xmlDepend)
        self.xmlContent.getroot().insert(0, xmlProject)

    class DependencyType(Enum):
        ALL      = ''
        EXTERNAL = 'ExtLib'
        INTERNAL = 'Lib'
        def isInternal(self):
            return self is self.ALL or self is self.INTERNAL
        def isExternal(self):
            return self is self.ALL or self is self.EXTERNAL
    def getProjectDependencies(self, gdkAppName, depType = DependencyType.ALL):
        dependencies = []
        self.log("Find project dependencies for " + gdkAppName)
        for xmlProject in self.xmlContent.findall("Project"):
            className = xmlProject.findtext('Classification')
            projType = xmlProject.get('type')
            if className:
                pType = Classification[className]
                isTool = pType.isTool()
            else:
                isTool = projType == 'tool'
            if isTool:
                projName = path.splitext(path.basename(xmlProject.get('name')))[0]
                if gdkAppName == projName:
                    if depType.isInternal():
                        dependencies.extend([xmlDepend.get('name') for xmlDepend in xmlProject.findall("Dependency")])
                    if depType.isExternal():
                        dependencies.extend([xmlDepend.get('name') for xmlDepend in xmlProject.findall("ExtDependency")])
                    break

        if dependencies:
            self.log("Found dependencies:")
            self.log(dependencies)
        else:
            self.log("No dependencies found.")
        return dependencies

    def getProjectClassification(self, gdkAppName):
        self.log("Find project type for " + gdkAppName)
        for xmlProject in self.xmlContent.findall("Project"):
            projName = path.splitext(path.basename(xmlProject.get('name')))[0]
            if gdkAppName == projName:
                return Classification[xmlProject.findtext('Classification')]
        raise Exception("Project {} not found".format(gdkAppName))

def getExternLibFiles(workDir, dependencyDir, platform):
    depPath = path.realpath(path.join(workDir, 'extern', dependencyDir, 'bin', platform.platName()))
    files = glob(path.join(depPath, '*'))
    return files

def createGdkPackage(workdir, gdkProjName, gocVersion, platforms, solution=None, log=print):
    log("Creating Gdk Package for {0} version {1}".format(gdkProjName, gocVersion))

    gdkAppName = 'GdkApp' + gdkProjName                    \
                                    .replace('GdkApp','')  \
                                    .replace('gdkapp','')  \
                                    .replace('Gdk','')     \
                                    .replace('GDK','')

    outdir = path.join(workdir, '..', 'pkg')
    if solution is not None:
         outdir = path.join(outdir, solution.getProjectClassification(gdkProjName).name) 
    tarPath = path.join(outdir, '{}.tar'.format(gdkAppName))
    os.makedirs(path.dirname(tarPath), exist_ok=True)
    with tarfile.TarFile(tarPath, 'w', format=tarfile.GNU_FORMAT) as tf:
        manifest = GdkManifest(gocVersion)
        internalDependencies = [] #solution.getProjectDependencies(gdkProjName, SolutionXml.DependencyType.INTERNAL) # Internal dependedncies are not needed in VE repo
        externalDependencies = solution.getProjectDependencies(gdkProjName, SolutionXml.DependencyType.EXTERNAL) if solution else []
        for plat in platforms:
            
            # Get the list of library files (including dependencies)
            # On windows, internal dependencies (Ghl, Gvl, TopoGeom) are considered external
            gdkAppFileNames = [("{GDKAPP}{EXT}".format(GDKAPP = gdkAppName, EXT=plat.ext()), "GdkApp")]
            if plat.isWin():
                libDlls = ["{LIBNAME}{SUFFIX}{EXT}".format(LIBNAME = libName, SUFFIX = "", EXT=plat.ext()) for libName in internalDependencies]
                gdkAppFileNames.extend([(libDll, "Lib") for libDll in libDlls])
            libFound = True
            libFileNames = []
            for libFileName, libFileType in gdkAppFileNames:
                libFilePath = glob(path.join(workdir, '..', 'bin', plat.platName(), libFileName))
                if not libFilePath:
                    libFilePath = glob(path.join(workdir, '..', 'lib', plat.platName(), libFileName))
                if libFilePath:
                    log("Adding {0} ; Modified {1}".format(path.join(plat.platName(), libFileName),
                                                           time.ctime(path.getmtime(libFilePath[0]))))
                    libFileNames.append((libFilePath[0], libFileType))
                else:
                    log("Warning: File Not Found: {0}. Platform {1} is incomplete.".format(libFileName, plat.platName()))
                    libFound = False
            
            if libFound:
                for extLib in externalDependencies:
                    libFileNames.extend([(extlib, "ExtLib") for extlib in getExternLibFiles(workdir, extLib, plat)])

                for libFilePath, libFileType in libFileNames:
                    libFileName = path.basename(libFilePath)
                    tf.add(libFilePath, path.join(plat.platName(), libFileName))
                    manifest.addFile(gdkAppName, plat, libFileType, path.join(plat.platName(), libFileName), plat.path())
        tf.addfile(*manifest.toTar(GDK_MANIFEST_XML))
    return gdkAppName, tarPath

def AddOnBinFiles(workdir, toolNames, classification):    
    '''Extract all the tar packages into the workdir's "bin" directory
    (linux_arm64 files are extracted into the "lib" directory instead)
    then returns a list of files from the "bin" and "lib" directories respectively'''
    binNames = {}
    libNames = {}
    for p in Platforms:
        binNames[p] = []
        libNames[p] = []

    binPath = path.join(workdir, 'bin')
    libPath = path.join(workdir, 'lib')
    for gdkAppName in toolNames:
        tarPath = path.join(workdir, 'pkg', classification.name, '{}.tar'.format(gdkAppName))
        if not path.isfile(tarPath):
            continue

        with tarfile.TarFile(tarPath) as tf:
            for platform in Platforms:
                if platform is Platforms.LINUX_ARM64:
                    extractPath = libPath
                else:
                    extractPath = binPath
                platformFiles = [
                    tarinfo for tarinfo in tf.getmembers()
                    if tarinfo.name.startswith(platform.platName() + '/')
                ]
                if not platformFiles:
                    continue
                tf.extractall(extractPath, platformFiles)
            manifestMember = tf.getmember(GDK_MANIFEST_XML)
            manifestFile = GdkManifest.fromString(tf.extractfile(manifestMember).read())

        for platform in Platforms:
            if platform is Platforms.LINUX_ARM64:
                libNames[platform].extend(manifestFile.getFiles(platform))
            else:    
                binNames[platform].extend(manifestFile.getFiles(platform))
    return binNames, libNames

def getGocatorVersion(workdir):
    version = ""
    datFiles = glob(path.join(workdir, '..', 'pkg', '*_SOFTWARE_*.dat'))
    if datFiles:            
        version_pattern = r"\d+\.\d+\.\d+\.\d+"
        m = re.findall(version_pattern, path.basename(datFiles[0]))
        if m:
            version = m[0]
        return version
    raise Exception("Couldn't get Gocator Version")

def Package(workdir, projectName, config):
    platforms = [
        Platforms.ARM7,
        Platforms.LINUX_ARM64
    ]
    if config.lower() == 'release':
        platforms.append(Platforms.WIN32)
        platforms.append(Platforms.WIN64)
    elif config.lower() == 'debug':
        platforms.append(Platforms.WIN32D)
        platforms.append(Platforms.WIN64D)
    else:
        raise Exception("Unrecognized config name '{}'".format(config))

    createGdkPackage(workdir, projectName, getGocatorVersion(workdir), platforms)

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Package GDK Tool as an Add On package.')
    parser.add_argument('project_dir', help='Path to Project directory.')
    parser.add_argument('config', help='Configuration name.', default='Release')
    args, unknown = parser.parse_known_args()

    project_dir = path.realpath(args.project_dir)
    workdir = path.realpath(path.join(project_dir, '..'))
    projectName = path.basename(project_dir)
    Package(workdir, projectName, args.config)