/**
* @file    GvProfileMsg.h
* @brief   Declares the GvProfileMsg class.
*
* Copyright (C) 2017-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GV_PROFILE_MSG_H
#define GV_PROFILE_MSG_H

#include <GoVision/Data/GvDataMsg.h>

/**
* Encapsulates a profile message.
*
* @class       GvProfileMsg
* @extends     GvDataMsg
* @ingroup     GoVision-Data
*/
typedef GvDataMsg GvProfileMsg;

GvFx(kStatus) GvProfileMsg_Construct(GvProfileMsg* msg, kType pointType, kSize count, kSize pointCount, kAlloc alloc);

/** 
 * Returns the message point type.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @return              Point type.
 */
GvFx(kType) GvProfileMsg_PointType(GvProfileMsg msg);

/** 
 * Returns the number of points.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @return              Point count.
 */
GvFx(kSize) GvProfileMsg_PointCount(GvProfileMsg msg);

/** 
 * Returns the x,y,z offset for the data message.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @return              Pointer to data offset.
 */
GvFx(const kPoint3d64f*) GvProfileMsg_Offset(GvProfileMsg msg);

/** 
 * Sets the x,y,z offset for the data message.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @param   offset      Pointer to data offset.
 * @return              Operation status.
 */
GvFx(kStatus) GvProfileMsg_SetOffset(GvProfileMsg msg, const kPoint3d64f* offset);

/** 
 * Returns the x,y,z scale of the data message.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @return              Pointer to data scale.
 */
GvFx(const kPoint3d64f*) GvProfileMsg_Scale(GvProfileMsg msg);

/** 
 * Sets the x,y,z scale of the data message.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @param   scale       Pointer to data scale.
 * @return              Operation status.
 */
GvFx(kStatus) GvProfileMsg_SetScale(GvProfileMsg msg, const kPoint3d64f* scale);

/** 
 * Returns a pointer to the points buffer.
 *
 * The buffer can be both read and written.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @return              Buffer pointer.
 */
GvFx(void*) GvProfileMsg_Points(GvProfileMsg msg);

/** 
 * Copies the points buffer with the data from an array to the message.
 *
 * The item type and size of the array must match those of the message.
 * Otherwise an error is returned.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @param   points      Points array.
 * @return              Operation status.
 */
GvFx(kStatus) GvProfileMsg_CopyPointsArray(GvProfileMsg msg, kArray1 points);

/** 
 * @deprecated. Use GvProfileMsg_CopyPointsArray() instead.
 * Updates the points buffer with the data from an array.
 *
 * The item type and size of the array must match those of the message.
 * Otherwise an error is returned.
 * Note: this does not transfer ownership of the passed array but only copies it 
 * into the message object.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @param   points      Points array.
 * @return              Operation status.
 */
GvFx(kStatus) GvProfileMsg_SetPointsArray(GvProfileMsg msg, kArray1 points);

/** 
 * Returns an array object containing the points.
 *
 * The user must not free the returned array.
 *
 * The implementation reuses the points buffer so it is fairly efficient.
 * A shell object is still constructed however so it is not as efficient as
 * using the buffer directly.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @return              Buffer pointer.
 */
GvFx(kArray1) GvProfileMsg_PointsArray(GvProfileMsg msg);

/** 
 * Returns whether or not this message contains intensity data.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @return              Presence of intensity data.
 */
GvFx(kBool) GvProfileMsg_HasIntensity(GvProfileMsg msg);

/** 
 * Returns the message intensity type.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @return              Intensity type.
 */
GvFx(kType) GvProfileMsg_IntensityType(GvProfileMsg msg);

/** 
 * Allocates an intensity buffer for the message.
 *
 * If the buffer already exists and is of the same type as requested, success
 * is returned. If it exists but type does not match, kERROR_ALREADY_EXISTS is returned.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @param   type        Type of intensity values.
 * @return              Operation status.
 */
GvFx(kStatus) GvProfileMsg_AllocateIntensity(GvProfileMsg msg, kType type);

/** 
 * Returns a pointer to the intensity buffer.
 *
 * The buffer can be both read and written.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @return              Buffer pointer.
 */
GvFx(void*) GvProfileMsg_Intensity(GvProfileMsg msg);

/** 
 * Copies the intensity buffer with the data from an array to the message.
 *
 * The item type and size of the array must match those of the message.
 * Otherwise an error is returned.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @param   intensity   Intensity array.
 * @return              Operation status.
 */
GvFx(kStatus) GvProfileMsg_CopyIntensityArray(GvProfileMsg msg, kArray1 intensity);

/** 
 * @deprecated. Use GvProfileMsg_CopyIntensityArray() instead.
 * Updates the intensity buffer with the data from an array.
 *
 * The item type and size of the array must match those of the message.
 * Otherwise an error is returned.
 * Note: this does not transfer ownership of the passed array but only copies it 
 * into the message object.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @param   intensity   Intensity array.
 * @return              Operation status.
 */
GvFx(kStatus) GvProfileMsg_SetIntensityArray(GvProfileMsg msg, kArray1 intensity);

/** 
 * Returns an array object containing the intensity data.
 *
 * The user must not free the returned array.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @return              Array object.
 */
GvFx(kArray1) GvProfileMsg_IntensityArray(GvProfileMsg msg);

/** 
 * Returns a pointer to the slices buffer.
 *
 * The buffer can be both read and written.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object. 
 * @return              Buffer pointer.
 */
GvFx(void*) GvProfileMsg_Slice(GvProfileMsg msg);

/**
 * Sets the rendering object within the message.
 *
 * @public               @memberof GvProfileMsg
 * @param    msg         Message object.
 * @param    object      GdkGraphic object (ownership is transferred).
 * @return               Operation status.
 */
GvFx(kStatus) GvProfileMsg_SetRendering(GvProfileMsg msg, kObject object);

/**
 * Retrieves the rendering object from the message.
 *
 * @public               @memberof GvProfileMsg
 * @param    msg         Message object.
 * @return               GdkGraphic object (ownership is transferred).
 */
GvFx(kObject) GvProfileMsg_Rendering(GvProfileMsg msg);

/**
 * Creates a table to store the starting location of each sub-profile
 * that makes up the profile in this profile message. 
 * The sub-profiles are profiles from buddy sensors.
 * The starting location of a sub-profile is the index value into
 * this profile message's merged profile where the sub-profile begins.
 *
 * Call this function if profile starting location/index information
 * is required and applicable for this profile message.
 * Scenarios where this is needed is for a buddy configuration
 * with at least one buddy sensor.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object.
 * @param   numEntries  Number of buddy sensors (main sensor is excluded).
 * @param   alloc       The memory allocator to use.
 * @return              Operation status.
 */
GvFx(kStatus) GvProfileMsg_ConstructStartTable(GvProfileMsg msg, kSize numEntries, kAlloc alloc);

/**
 * Add the starting location/index for a profile into the profile message's
 * profile start table.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object.
 * @param   tableEntry  Location in the start table to store the 
 *                      starting location of the buddy profile.
 * @param   startValue  Starting location of a buddy sensor's profile.
 * @return              Operation status.
 */
GvFx(kStatus) GvProfileMsg_AddIndexToStartTable(GvProfileMsg msg, kSize tableEntry, k32u startValue);

/**
 * Return handle to the profile message's profile start table.
 *
 * @public              @memberof GvProfileMsg
 * @param   msg         Message object.
 * @return              kArray1 handle for the profile start table.
 */
GvFx(kArray1) GvProfileMsg_StartTable(GvProfileMsg msg);

#include <GoVision/Data/GvProfileMsg.x.h>

#endif
