/** 
 * @file    TestProfileSelect.c
 *
 * Copyright © 2015-2022 by LMI Technologies Inc.  All rights reserved.
 */
#include <GdkAppSample/TestProfileSelect.h>

kBeginClassEx(Tool, TestProfileSelect)
    kAddVMethod(TestProfileSelect, kObject, VRelease)
    kAddVMethod(TestProfileSelect, GdkTool, VInit)
    kAddVMethod(TestProfileSelect, GdkTool, VDescribe)
    kAddVMethod(TestProfileSelect, GdkTool, VUpdateConfig)
    kAddVMethod(TestProfileSelect, GdkTool, VStart)
    kAddVMethod(TestProfileSelect, GdkTool, VStop)
    kAddVMethod(TestProfileSelect, GdkTool, VProcess)
kEndClassEx()

ToolFx(kStatus) TestProfileSelect_VDescribe(GdkToolInfo toolInfo)
{
    GdkMeasurementInfo mmtInfo;
    GdkRegionXZ64f defRegion = { -35, -30, 70, 60 };
    kBool defBool = kTRUE;
    k32s defInt = 0;
    
    kCheck(GdkToolInfo_SetTypeName(toolInfo, TEST_PROFILE_SELECT_TOOL_NAME));
    kCheck(GdkToolInfo_SetLabel(toolInfo, TEST_PROFILE_SELECT_TOOL_LABEL));
    kCheck(GdkToolInfo_EnableAutoVersion(toolInfo, kFALSE));

    kCheck(GdkToolInfo_SetSourceType(toolInfo, GDK_DATA_TYPE_UNIFORM_PROFILE));

    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP));
    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP_LEFT));
    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP_RIGHT));

    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_X, kTRUE));
    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_Z, kTRUE));

    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_BOOL, "UseRegion", "Use Region", &defBool, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_PROFILE_REGION, "Region", "Region", &defRegion, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_INT, "Index", "Index", &defInt, kNULL));
    
    // Create measurement. Must the same order as measurement index!

    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "X", "Position X", &mmtInfo));
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_X));
      
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, "Z", "Position Z", &mmtInfo));
    kCheck(GdkMeasurementInfo_SetValueType(mmtInfo, GDK_MEASUREMENT_VALUE_TYPE_Z));
    
    return kOK;
}

ToolFx(kStatus) TestProfileSelect_VInit(TestProfileSelect tool, kType type, kAlloc alloc)
{
    kObjR(TestProfileSelect, tool);

    kCheck(GdkTool_VInit(tool, type, alloc));
    kZero(obj->dataSource);
    obj->useRegion = kFALSE;
    kZero(obj->region);
    obj->index = 0;

    return kOK;
}

ToolFx(kStatus) TestProfileSelect_VRelease(TestProfileSelect tool)
{
    kObj(TestProfileSelect, tool);

    return GdkTool_VRelease(tool);
}

ToolFx(kStatus) TestProfileSelect_VUpdateConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    GdkParams params = GdkToolCfg_Parameters(toolConfig);
    GdkParam paramUseRegion = GdkParams_Find(params, "UseRegion");
    GdkParam paramRegion = GdkParams_Find(params, "Region");

    kCheck(GdkParam_SetUsed(paramRegion, GdkParam_AsBool(paramUseRegion)));
    
    return kOK;
}

ToolFx(kStatus) TestProfileSelect_VStart(TestProfileSelect tool)
{
    kObj(TestProfileSelect, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkParams params = GdkToolCfg_Parameters(config);
    GdkParam param;

    obj->dataSource = GdkToolCfg_Source(config);
    
    param = GdkParams_Find(params, "UseRegion");
    obj->useRegion = GdkParam_AsBool(param);

    param = GdkParams_Find(params, "Region");
    obj->region = *GdkParam_AsProfileRegion(param);

    param = GdkParams_Find(params, "Index");
    obj->index = GdkParam_AsInt(param);

    return kOK;
}

ToolFx(kStatus) TestProfileSelect_VStop(TestProfileSelect tool)
{
    return kOK;
}

ToolFx(kStatus) TestProfileSelect_OutputPoint(TestProfileSelect tool, k32u measurementIndex, const kPoint3d64f *point, GdkToolOutput output)
{
    kObj(TestProfileSelect, tool);
    GdkGraphic graphic = kNULL;
    GdkGraphicPointSet pointSet = kNULL;
    kStatus exception = kOK;
    kPoint3d32f point32f;
    kAlloc alloc;

    alloc = GdkTool_MessageAlloc(tool);

    // Convert 64f point to 32f. Graphics only supprot 32f data
    if (point->x != k64F_NULL) point32f.x = (k32f)point->x;
    else point32f.x = k32F_NULL;
    
    if (point->y != k64F_NULL) point32f.y = (k32f)point->y;
    else point32f.y = k32F_NULL;

    if (point->z != k64F_NULL) point32f.z = (k32f)point->z;
    else point32f.z = k32F_NULL;

    kTry
    {
        kTest(GdkGraphic_Construct(&graphic, alloc));
        kTest(GdkGraphicPointSet_Construct(&pointSet, 4.0, kMARKER_SHAPE_CROSS, kCOLOR_LIME, &point32f, 1, alloc));
        kTest(GdkGraphic_AddPointSet(graphic, pointSet));
        pointSet = kNULL;

        kTest(GdkToolOutput_SetRendering(output, measurementIndex, graphic));
        graphic = kNULL;
    }
    kCatchEx(&exception)
    {
        kEndCatchEx(exception);
    }
    kFinallyEx
    {
        kDestroyRef(&pointSet);
        kEndFinallyEx();
    }
    return kOK;
}

ToolFx(kStatus) TestProfileSelect_OutputValue(GdkToolCfg config, kSize index, k64f value, GdkMeasurementDecision decision, GdkToolOutput output)
{
    GvMeasureMsg msg = kNULL;

    if (GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, index)))
    {
        kCheck(GdkToolOutput_InitMeasurementAt(output, index, &msg));
        if (msg != kNULL)
        {
            kCheck(GvMeasureMsg_SetValue(msg, value));
            kCheck(GvMeasureMsg_SetStatus(msg, decision));
        }
    }

    return kOK;
}

ToolFx(kStatus) TestProfileSelect_VProcess(TestProfileSelect tool, GdkToolInput input, GdkToolOutput output)
{
    kObj(TestProfileSelect, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkInputItem item;
    GdkDataInfo itemInfo;
    const kPoint3d64f* anchor = GdkToolInput_AnchorPosition(input);
    GdkRegionXZ64f offsetRegion = { k64F_NULL, k64F_NULL, k64F_NULL, k64F_NULL };
    kStatus exception = kOK;

    item = GdkToolInput_Find(input, obj->dataSource);
    if (!item) return kERROR_PARAMETER;

    itemInfo = GdkInputItem_Info(item);

    kTry
    {

        k32u regionStartIndex, regionEndIndex, dataIndex;

        if(obj->useRegion)
        {
            // Adjust user region based on anchoring information. If anchoring is not enabled, anchor is zero.
            offsetRegion = obj->region;
            offsetRegion.x += anchor->x;
            offsetRegion.z += anchor->z;
        }
        else
        {
            // If user didn't select the region, use the maximum region from the input data. 
            offsetRegion.x = GdkInputItem_Offset(item)->x;
            offsetRegion.width = GdkProfileInput_Count(item) * GdkDataInfo_Scale(itemInfo)->x;
            offsetRegion.z = GdkDataInfo_Region(itemInfo)->z;
            offsetRegion.height = GdkDataInfo_Region(itemInfo)->height;
        }

        regionStartIndex = (k32s)((offsetRegion.x - GdkInputItem_Offset(item)->x) / GdkDataInfo_Scale(itemInfo)->x); // index where the region starts 
        regionEndIndex = (k32s)(((offsetRegion.x + offsetRegion.width) - GdkInputItem_Offset(item)->x) / GdkDataInfo_Scale(itemInfo)->x); // index where the region ends
        
        dataIndex = obj->index + regionStartIndex;

        if ((dataIndex > GdkProfileInput_Count(item)) || (obj->index > (regionEndIndex - regionStartIndex + 1)))
        {
            kTest(TestProfileSelect_OutputValue(config, TEST_PROFILE_SELECT_MEASUREMENT_X, 0.0, GDK_MEASUREMENT_ERROR_VALUE, output));
            kTest(TestProfileSelect_OutputValue(config, TEST_PROFILE_SELECT_MEASUREMENT_Z, 0.0, GDK_MEASUREMENT_ERROR_VALUE, output));
        }
        else
        {
            kPoint3d64f point;

            const k16s *profileData = GdkProfileInput_Ranges(item);    

            point.x = (regionStartIndex + obj->index) * GdkDataInfo_Scale(itemInfo)->x + GdkInputItem_Offset(item)->x;;
            point.y = 0;            

            kTest(TestProfileSelect_OutputValue(config, TEST_PROFILE_SELECT_MEASUREMENT_X, point.x, GDK_MEASUREMENT_OK, output));

            if (profileData[dataIndex] != k16S_NULL)
            {
                point.z = profileData[dataIndex] * GdkDataInfo_Scale(itemInfo)->z + GdkInputItem_Offset(item)->z;                
                kTest(TestProfileSelect_OutputValue(config, TEST_PROFILE_SELECT_MEASUREMENT_Z, point.z, GDK_MEASUREMENT_OK, output));
                kTest(TestProfileSelect_OutputPoint(tool, TEST_PROFILE_SELECT_MEASUREMENT_X, &point, output));
                kTest(TestProfileSelect_OutputPoint(tool, TEST_PROFILE_SELECT_MEASUREMENT_Z, &point, output));
            }                
            else 
            {
                // No z data at that point
                point.z = k32F_NULL;
                kTest(TestProfileSelect_OutputValue(config, TEST_PROFILE_SELECT_MEASUREMENT_Z, k16S_NULL, GDK_MEASUREMENT_ERROR_VALUE, output));
            }            
        }                       
    }
    kCatchEx(&exception)
    {
        kEndCatchEx(exception);
    }
    kFinallyEx
    {
        kEndFinallyEx();
    }

    return kOK;
}

