/////////////////////////////////////////////////////////////////////////////
// TestProfileDynamicAlignment
// 2017-02-01 (C) LMI Technologies
/////////////////////////////////////////////////////////////////////////////

#include <GdkAppSample/TestProfileDynamicAlignment.h>
#include <kVision/Vs/kVsUtilities.h>

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

kBeginClassEx(Tool, TestProfileDynamicAlignment)
    kAddVMethod(TestProfileDynamicAlignment, kObject, VRelease)
    kAddVMethod(TestProfileDynamicAlignment, GdkTool, VInit)
    kAddVMethod(TestProfileDynamicAlignment, GdkTool, VName)
    kAddVMethod(TestProfileDynamicAlignment, GdkTool, VDescribe)
    kAddVMethod(TestProfileDynamicAlignment, GdkTool, VNewMeasurementConfig)
    kAddVMethod(TestProfileDynamicAlignment, GdkTool, VUpdateConfig)
    kAddVMethod(TestProfileDynamicAlignment, GdkTool, VNewToolConfig)
    kAddVMethod(TestProfileDynamicAlignment, GdkTool, VStart)
    kAddVMethod(TestProfileDynamicAlignment, GdkTool, VStop)
    kAddVMethod(TestProfileDynamicAlignment, GdkTool, VProcess)
kEndClassEx()

/////////////////////////////////////////////////////////////////////////////
// GtTool functions
/////////////////////////////////////////////////////////////////////////////

ToolFx(const kChar*) TestProfileDynamicAlignment_VName()
{
    return TEST_PROFILE_DYNAMIC_ALIGNMENT_TOOL_NAME;
}

ToolFx(kStatus) TestProfileDynamicAlignment_VDescribe(GdkToolInfo toolInfo)
{
    GdkToolDataOutputInfo toolDataInfo = kNULL;
    GdkMeasurementInfo measInfo = kNULL;
    GdkParamInfo paramInfo = kNULL;
    GdkRegionXZ64f defaultRegion = { -10, -10, 20, 20 };
    kSize defaultSelection = 1;

    kCheck(GdkToolInfo_SetLabel(toolInfo, TEST_PROFILE_DYNAMIC_ALIGNMENT_TOOL_LABEL));
    kCheck(GdkToolInfo_SetTypeName(toolInfo, TEST_PROFILE_DYNAMIC_ALIGNMENT_TOOL_NAME));
    kCheck(GdkToolInfo_EnableAutoVersion(toolInfo, kFALSE));

    kCheck(GdkToolInfo_SetSourceType(toolInfo, GDK_DATA_TYPE_UNIFORM_PROFILE));

    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_X, kTRUE));
    kCheck(GdkToolInfo_EnableAnchoring(toolInfo, GDK_ANCHOR_PARAM_Z, kTRUE));

    kCheck(GdkToolInfo_AddSourceOption(toolInfo, GDK_DATA_SOURCE_TOP));

    // Input Parameters
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_INT, PARAM_NUM_REGION_NAME, PARAM_NUM_REGION_LABEL, &defaultSelection, &paramInfo));
    kCheck(GdkParamInfo_AddOptionInt(paramInfo, 1, "1"));
    kCheck(GdkParamInfo_AddOptionInt(paramInfo, 2, "2"));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_PROFILE_REGION, PARAM_REGION1_NAME, PARAN_REGION1_LABEL, &defaultRegion, kNULL));
    kCheck(GdkToolInfo_AddParam(toolInfo, GDK_PARAM_TYPE_PROFILE_REGION, PARAM_REGION2_NAME, PARAN_REGION2_LABEL, &defaultRegion, kNULL));

    // Output Measurements
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, MEAS_ANGLE_NAME, MEAS_ANGLE_LABEL, &measInfo));                    // #0
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_MEASUREMENT, MEAS_INTERCEPT_NAME, MEAS_INTERCEPT_LABEL, &measInfo));            // #1
    kCheck(GdkToolInfo_AddOutput(toolInfo, GDK_DATA_TYPE_UNIFORM_PROFILE, OUTPUT_ALIGNED_NAME, OUTPUT_ALIGNED_LABEL, &toolDataInfo));    // #2

    return kOK;
}

ToolFx(kStatus) TestProfileDynamicAlignment_VInit(TestProfileDynamicAlignment tool, kType type, kAlloc alloc)
{
    kObjR(TestProfileDynamicAlignment, tool);

    kCheck(GdkTool_VInit(tool, type, alloc));
    kZero(obj->dataSource);
    kZero(obj->region1);
    kZero(obj->region2);

    return kOK;
}

ToolFx(kStatus) TestProfileDynamicAlignment_VRelease(TestProfileDynamicAlignment tool)
{
    kObj(TestProfileDynamicAlignment, tool);

    return GdkTool_VRelease(tool);
}

ToolFx(kStatus) TestProfileDynamicAlignment_VNewToolConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    return kOK;
}

ToolFx(kStatus) TestProfileDynamicAlignment_VNewMeasurementConfig(const GdkToolEnv* env, GdkToolCfg toolConfig, GdkMeasurementCfg measurementConfig)
{
    return kOK;
}

ToolFx(kStatus) TestProfileDynamicAlignment_VUpdateConfig(const GdkToolEnv* env, GdkToolCfg toolConfig)
{
    GdkParams params = GdkToolCfg_Parameters(toolConfig);
    GdkParam  paramNumRegions = GdkParams_Find(params, PARAM_NUM_REGION_NAME);
    GdkParam  paramRegion2    = GdkParams_Find(params, PARAM_REGION2_NAME);

    kBool useRegion2 = GdkParam_AsInt(paramNumRegions) == 2;
    kCheck(GdkParam_SetUsed(paramRegion2, useRegion2));

    return kOK;
}

ToolFx(kStatus) TestProfileDynamicAlignment_VStart(TestProfileDynamicAlignment tool)
{
    kObj(TestProfileDynamicAlignment, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    obj->dataSource = GdkToolCfg_Source(config);

    GdkParams params = GdkToolCfg_Parameters(config);

    obj->region1 = *GdkParam_AsProfileRegion(GdkParams_Find(params, PARAM_REGION1_NAME));
    if (GdkParam_AsInt(GdkParams_Find(params, PARAM_NUM_REGION_NAME)) == 2)
    {
        obj->region2 = *GdkParam_AsProfileRegion(GdkParams_Find(params, PARAM_REGION2_NAME));
    }
    else
    {
        obj->region2.x = 0;
        obj->region2.z = 0;
        obj->region2.width = 0;
        obj->region2.height = 0;
    }
    return kOK;
}

ToolFx(kStatus) TestProfileDynamicAlignment_VStop(TestProfileDynamicAlignment tool)
{
    kObj(TestProfileDynamicAlignment, tool);
    return kOK;
}

ToolFx(kStatus) TestProfileDynamicAlignment_VProcess(TestProfileDynamicAlignment tool, GdkToolInput input, GdkToolOutput output)
{
    kObj(TestProfileDynamicAlignment, tool);
    GdkToolCfg config = GdkTool_Config(tool);
    GdkSurfaceInput item = kNULL;
    GdkDataInfo itemInfo = kNULL;
    kSize columns = 0, newColumns = 0;
    const kPoint3d64f* anchor = GdkToolInput_AnchorPosition(input);
    kSize i = 0;
    k16s  k = 0;
    const kPoint3d64f* offset = kNULL, *scale = kNULL;
    kPoint3d64f newOffset = { 0 };
    kPoint64f pt = { 0, 0 }, newPt = { 0, 0 };
    kPoint16s indexPt = { 0, 0 };
    const k16s* pixel = kNULL;

    kArrayList regionPoints = kNULL;
    kL3dTransform2d transform = { 0 };
    k64f slope = k64F_NULL, intercept = k64F_NULL, angle = k64F_NULL;

    GvProfileMsg profileMsg = kNULL;
    kArray1 pointArray = kNULL;
    kArrayList mmPointArray = kNULL;

    item = GdkToolInput_Find(input, obj->dataSource);
    if (!item)
    {
        return kERROR_PARAMETER;
    }
    itemInfo = GdkInputItem_Info(item);

    kTry
    {
        columns = GdkProfileInput_Count(item);
        offset = GdkInputItem_Offset(item);
        scale = GdkDataInfo_Scale(itemInfo);
        newOffset = *offset;

        // Adjust user region based on anchoring information. If anchoring is not enabled, anchor is zero.
        obj->region1.x += anchor->x;
        obj->region1.z += anchor->z;
        obj->region2.x += anchor->x;
        obj->region2.z += anchor->z;

        // First extract all the points from the Regions of Interest
        kTest(kArrayList_Construct(&regionPoints, kTypeOf(kPoint64f), 0, kNULL));
        pixel = GdkProfileInput_Ranges(item);
        for (i = 0; i < columns; i++)
        {
            k = pixel[i];

            // For every point who is in range
            if (k != k16S_NULL)
            {
                pt.x = i*scale->x + offset->x;
                pt.y = k*scale->z + offset->z;

                // If the point is in the ROI
                if((pt.x > obj->region1.x && pt.x < obj->region1.x + obj->region1.width &&
                    pt.y > obj->region1.z && pt.y < obj->region1.z + obj->region1.height) ||
                   (obj->region2.height*obj->region2.width > 0 && // No second region
                    pt.x > obj->region2.x && pt.x < obj->region2.x + obj->region2.width &&
                    pt.y > obj->region2.z && pt.y < obj->region2.z + obj->region2.height))
                {
                    // Add it to the list
                    kTest(kArrayList_AddT(regionPoints, &pt));
                }
            }
        }

        // Create the transformation (only if there are enough points in the ROI)
        kTest(kL3dTransform2d_Identity(&transform));
        if (kArrayList_Count(regionPoints) > MIN_POINTS)
        {
            // Fit at line through all the points
            kTest(kVsLine2d_FitPoint64f(kArrayList_DataT(regionPoints, kPoint64f), (k32u)kArrayList_Count(regionPoints), &slope, &intercept));
            angle = atan(slope)*180/M_PI;

            // Subtract the angle and intercept point of the fitted line from the transformation matrix
            kTest(kL3dTransform2d_Rotate(&transform, -angle, &transform));
            kTest(kL3dTransform2d_Translate(&transform, 0, -intercept, &transform));
        }


        // Preform the transformation on all the profile points
        kTest(kArrayList_Construct(&mmPointArray, kTypeOf(kPoint64f), columns, kNULL));
        pixel = GdkProfileInput_Ranges(item);
        for (i = 0; i < columns; i++)
        {
            k = pixel[i];

            if (k != k16S_NULL)
            {
                // Switch to real-world coordinates (mm)
                pt.x = i*scale->x + offset->x;
                pt.y = k*scale->z + offset->z;

                // Apply transformation
                kTest(kL3dTransform2d_Apply(&transform, pt.x, pt.y, &newPt.x, &newPt.y));

                kTest(kArrayList_AddT(mmPointArray, &newPt));

                // Check if the offset should be changed after the rotation
                newOffset.x = kMin_(newPt.x, newOffset.x);
                newOffset.z = kMin_(newPt.y, newOffset.z);
            }
        }

        // Find the new number of points
        for (i = 0; i < kArrayList_Count(mmPointArray); i++)
        {
            newPt = *kArrayList_AtT(mmPointArray, i, kPoint64f);
            k16s x = kMath_Round16s_((newPt.x - newOffset.x) / scale->x);
            newColumns = kMax_(newColumns, (kSize)(x + 1));
        }

        // Initialize the output profile array
        kTest(kArray1_Construct(&pointArray, kTypeOf(k16s), newColumns, kNULL));
        k16s nullItem = k16S_NULL;
        for (i = 0; i < newColumns; i++)
        {
            kTest(kArray1_SetItemT(pointArray, i, &nullItem));
        }
        for (i = 0; i < kArrayList_Count(mmPointArray); i++)
        {
            newPt = *kArrayList_AtT(mmPointArray, i, kPoint64f);

            // Switch back to grid space coordinates
            indexPt.x = kMath_Round16s_((newPt.x - newOffset.x) / scale->x);
            indexPt.y = newPt.y == k64F_NULL? k16S_NULL: kMath_Round16s_((newPt.y - newOffset.z) / scale->z);

            // Add the point to the output array
            kTest(kArray1_SetItemT(pointArray, indexPt.x, &indexPt.y));
        }

        // Create the tool output message
        kTest(GdkToolOutput_InitProfileAt(output, OUTPUT_ALIGNED_INDEX, kArray1_Count(pointArray), &profileMsg));
        if (profileMsg != kNULL)
        {
            // Set the scale and the offset of the output message, and attach the point array
            kTest(GvProfileMsg_SetOffset(profileMsg, &newOffset));
            kTest(GvProfileMsg_SetScale(profileMsg, scale));

            kTest(GvProfileMsg_SetPointsArray(profileMsg, pointArray));
        }

        // Output the angle and intercept as measurements
        kTest(TestProfileDynamicAlignment_OutputValue(output, MEAS_ANGLE_INDEX, angle, angle != k64F_NULL, config));
        kTest(TestProfileDynamicAlignment_OutputValue(output, MEAS_INTERCEPT_INDEX, intercept, intercept != k64F_NULL, config));
    }
    kFinally
    {
        kDestroyRef(&regionPoints);
        kDestroyRef(&mmPointArray);
        kDestroyRef(&pointArray);

        kEndFinally();
    }

    return kOK;
}

/////////////////////////////////////////////////////////////////////////////
// Output rendering
/////////////////////////////////////////////////////////////////////////////

ToolFx(kStatus) TestProfileDynamicAlignment_OutputValue(GdkToolOutput output, kSize index, k64f value, kBool valid, GdkToolCfg config)
{
    GvMeasureMsg msg = kNULL;

    if (GdkMeasurementCfg_Enabled(GdkToolCfg_MeasurementAt(config, index)))
    {
        kCheck(GdkToolOutput_InitMeasurementAt(output, index, &msg));
        if (msg != kNULL)
        {
            if (valid)
            {
                kCheck(GvMeasureMsg_SetValue(msg, value));
                kCheck(GvMeasureMsg_SetStatus(msg, GV_MEASUREMENT_OK));
            }
            else
            {
                kCheck(GvMeasureMsg_SetStatus(msg, GV_MEASUREMENT_ERROR_INVALID_RESULT));
            }
        }
    }

    return kOK;
}
