/**
* @file    GdkToolOutput.h
* @brief   Declares the GdkToolOutput class.
*
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GDK_TOOL_OUTPUT_H
#define GDK_TOOL_OUTPUT_H

#include <Gdk/GdkDef.h>
#include <Gdk/Data/GdkGraphic.h>
#include <Gdk/Tools/GdkFeature.h>
#include <../GoVision/GoVision/Data/GvMeasureMsg.h>
#include <../GoVision/GoVision/Data/GvFeatureMsg.h>
#include <../GoVision/GoVision/Data/GvProfileMsg.h>
#include <../GoVision/GoVision/Data/GvSurfaceMsg.h>
#include <../GoVision/GoVision/Data/GvGenericMsg.h>
#include <../GoVision/GoVision/Data/GvMeshMsg.h>

/**
* Represents a tool output collection.
*
* @class       GdkToolOutput
* @extends     kObject
* @ingroup     Gdk-Tools
*/
typedef kObject GdkToolOutput;
kDeclareClassEx(Gdk, GdkToolOutput, kObject)
kDeclareValueEx(Gdk, GdkToolOutputItem, kValue)
kDeclareValueEx(Gdk, GdkToolDiagnosticOutputItem, kValue)


/**
* Sets the measurement rendering at a specified index.
*
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       Measurement index within the tool.
* @param    graphic     GdkGraphic object (ownership is transferred).
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolOutput_SetRendering(GdkToolOutput output, kSize index, GdkGraphic graphic);

/**
* Sets the diagnostic output at a specified index.
*
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       diagnostic output index within the tool. (The place in order the diagnostic output was added to the tool).
* @param    msg         An object that implements kMsgSet (ownership is transferred).
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolOutput_SetDiagnostic(GdkToolOutput output, kSize index, kMsgSet msg);

/**
* Retrieves the GvMeasureMsg object for the measurement at the specified index.
*
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       Index of the output object to retrieve relative to all outputs
* @param    msg         Pointer to where the GvMeasureMsg is to be set
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolOutput_InitMeasurementAt(GdkToolOutput output, kSize index, GvMeasureMsg* msg);

/**
* Retrieves the GvFeatureMsg object for the feature at the specified index.
*
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       Index of the output object to retrieve relative to all outputs
* @param    msg         Pointer to where the GvFeatureMsg-based class object is to be set
*                       This can be one of the following:
*                       GvPointFeatureMsg
*                       GvLineFeatureMsg
*                       GvCircleFeatureMsg
*                       GvPlaneFeaturemsg
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolOutput_InitFeatureAt(GdkToolOutput output, kSize index, GvFeatureMsg* msg);

/**
* Constructs/reuses a GvProfileMsg object at the specified index.
* If the indicated index is a #GDK_DATA_TYPE_UNIFORM_PROFILE, the profile array will be of type k16s
* If the indicated index is a #GDK_DATA_TYPE_PROFILE_POINT_CLOUD, the profile array will be of type kPoint16s
*
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       Index of the output object to retrieve relative to all outputs
* @param    pointCount  Count of points for this profile
* @param    msg         Pointer to where the GvProfileMsg is to be set
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolOutput_InitProfileAt(GdkToolOutput output, kSize index, kSize pointCount, GvProfileMsg* msg);

/**
* Constructs a GvSurfaceMsg object for a Uniform surface at the specified index.
*
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       Index of the output object to retrieve relative to all outputs
* @param    width       Width of the surface
* @param    height      Height of the surface
* @param    msg         Pointer to where the GvSurfaceMsg is to be set (PointType = k16s)
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolOutput_InitSurfaceAt(GdkToolOutput output, kSize index, kSize width, kSize height, GvSurfaceMsg* msg);

/**
* Constructs a GvSurfaceMsg object for a raw surface at the specified index.
*
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       Index of the output object to retrieve relative to all outputs
* @param    width       Width of the surface
* @param    height      Height of the surface
* @param    msg         Pointer to where the GvSurfaceMsg is to be set (PointType = kPoint3d16s)
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolOutput_InitRawSurfaceAt(GdkToolOutput output, kSize index, kSize width, kSize height, GvSurfaceMsg* msg);

/**
* Constructs/reuses a GvGenericMsg object at the specified index.
*
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       Index of the output object to retrieve relative to all outputs
* @param    isObject    Flag indicating is the generic data is a kObject or a data buffer
* @param    dataSize    Size of the generic data
* @param    msg         Pointer to where the GvGenericMsg is to be set
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolOutput_InitGenericDataAt(GdkToolOutput output, kSize index, kBool isObject, kSize dataSize, GvGenericMsg* msg);

/**
* Constructs/reuses a GvMeshMsg object at the specified index.
*
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       Index of the output object to retrieve relative to all outputs
* @param    msg         Pointer to where the GvMeshMsg is to be set
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolOutput_InitMeshAt(GdkToolOutput output, kSize index, GvMeshMsg* msg);

//// Deprecated API functions

/**
* [Deprecated] Use GvMeasureMsg_SetValue(), GvMeasureMsg_SetStatus() instead.
*
* Sets the measurement output at a specified index.
*
* @deprecated
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       Measurement index within the tool.
* @param    value       Output value.
* @param    decision    Output decision.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolOutput_SetResult(GdkToolOutput output, kSize index, k64f value, GdkMeasurementDecision decision);

/**
* [Deprecated] Use GvMeasureMsg_SetResultPosition() instead.
*
* If the measurement value is a component of the position of a feature, use this
* function to associate the full position with the measurement.
*
* For a measurement that may be used as an anchor input to other tools, the
* position of the associated feature should be provided so downstream tools
* can use it for visualization or more advanced processing techniques.
*
* @deprecated
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       Measurement index within the tool.
* @param    position    Position.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolOutput_SetResultPosition(GdkToolOutput output, kSize index, const kPoint3d64f* position);

/**
* [Deprecated] Use GdkToolOutput_InitFeatureAt() instead.
*
* Gets the feature class object at the specified index.
*
* @deprecated
* @public               @memberof GdkToolOutput
* @param    output      Output object.
* @param    index       Index of the output object to retrieve.
* @return               GdkFeature class object.
*/
GdkFx(GdkFeature) GdkToolOutput_FeatureAt(GdkToolOutput output, kSize index);

#include <Gdk/Tools/GdkToolOutput.x.h>

#endif 
