/**
* @file    GdkToolInfo.x.h
*
* @internal
* Copyright (C) 2016-2022 by LMI Technologies Inc.
* Licensed under the MIT License.
* Redistributed files must retain the above copyright notice.
*/
#ifndef GDK_TOOL_INFO_X_H
#define GDK_TOOL_INFO_X_H

#define GdkToolInfo_NoSourceTypesSet(GdkToolInfo_info) \
        (GdkToolInfo_SourceTypesCount(GdkToolInfo_info) == 0 || GdkToolInfo_HasSourceType(GdkToolInfo_info, GDK_DATA_TYPE_NONE))

/**
* Marks the tool as an internal (LMI-Developed) tool
*
* @private              @memberof GdkToolInfo
* @param    info        info object
* @return               Operation status
*/
GdkFx(kStatus) GdkToolInfo_SetInternal(GdkToolInfo info);

/**
* Returns True if the tool is an internal (LMI-Developed) tool
*
* @private              @memberof GdkToolInfo
* @param    info        info object
* @return               Is the tool internal?
*/
GdkFx(kBool) GdkToolInfo_IsInternal(GdkToolInfo info);

/**
* Sets whether or not this tool requires intensity data.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    required    Whether or not intensity is required.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_RequireIntensity(GdkToolInfo info, kBool required);

/**
* Returns whether or not this tool requires intensity data.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Intensity requirement.
*/
GdkFx(kBool) GdkToolInfo_IntensityRequired(GdkToolInfo info);

/**
* Returns the number of acceptable data sources.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Data source count.
*/
GdkFx(kSize) GdkToolInfo_SourceOptionCount(GdkToolInfo info);

/**
* Returns the data source option at a specified index.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    index       Index of data source.
* @return               Data source.
*/
GdkFx(GdkDataSource) GdkToolInfo_SourceOptionAt(GdkToolInfo info, kSize index);

/**
* Returns whether or not any of the input data type parameters for this tool match the type passed.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    type        Data type to match.
* @return               True if match found, False otherwise.
*/
GdkFx(kBool) GdkToolInfo_HasSourceType(GdkToolInfo info, GdkDataType type);

/**
* Returns the number of data type parameters are set as inputs for this tool.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Number of source types set.
*/
GdkFx(kSize) GdkToolInfo_SourceTypesCount(GdkToolInfo info);

/**
* Returns the array list containing the data type parameters set as inputs for this tool.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Number of source types set.
*/
GdkFx(kArrayList) GdkToolInfo_SourceTypes(GdkToolInfo info);

/**
* Returns the measurement type at a specified index.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    index       Measurement type index.
* @return               Measurement type.
*/
GdkFx(GdkMeasurementInfo) GdkToolInfo_MeasurementAt(GdkToolInfo info, kSize index);

/**
* Clears measurement types.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_ClearMeasurements(GdkToolInfo info);

/**
* Returns the feature type at a specified index.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    index       Feature type index.
* @return               Feature type.
*/
GdkFx(GdkFeatureInfo) GdkToolInfo_FeatureAt(GdkToolInfo info, kSize index);

/**
* Clears feature types.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_ClearFeatures(GdkToolInfo info);

/**
* Returns the info of the tool data output with the specified index.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    index       Tool data output index.
* @return               Tool data output Info
*/
GdkFx(GdkToolDataOutputInfo) GdkToolInfo_ToolDataOutputAt(GdkToolInfo info, kSize index);

/**
* Clears the list of acceptable data sources.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_ClearSourceOptions(GdkToolInfo info);

/**
* Finds version object by name.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    name        Version name (can be empty string).
* @return               Found version object or kNULL.
*/
GdkFx(GdkToolVersionInfo) GdkToolInfo_FindVersion(GdkToolInfo info, const kChar* name);

/**
* Returns the first version info object. This object is implicitly added and is
* used by GdkToolInfo methods that access version information directly.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Version info object.
*/
GdkFx(GdkToolVersionInfo) GdkToolInfo_FirstVersion(GdkToolInfo info);

/**
* Convenience method for setting the name of the first version.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    name        Version name.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_SetFirstVersionName(GdkToolInfo info, const kChar* name);

/**
* Marks the tool as compatible with multiple frames of references.
*
* *** EXPERIMENTAL FEATURE - Will change in a future release ***
*
* A multi-reference frame compatible tool is able to use the reference frame information
* contained in the inputs to transform the result into a common frame of reference.
* For more information please see the section on coordinate systems.
*
* @public                   @memberof GdkToolInfo
* @param    info            info object
* @param    enabled         Enable multi-reference frame compatibility.
* @return                   Operation status
*/
GdkFx(kStatus) GdkToolInfo_SetMultiRefFrameCompatible(GdkToolInfo info, kBool enabled);

/**
* Returns if the tool is compatible with multiple frames of reference
*
* *** EXPERIMENTAL FEATURE - Will change in a future release ***
*
* @public               @memberof GdkToolInfo
* @param    info        info object
* @return               Multi-reference frame compatibility.
*/
GdkFx(kBool) GdkToolInfo_MultiRefFrameCompatible(GdkToolInfo info);

/**
* Adds a Diagnostic output to a tool with the specified index

* @public                   @memberof GdkToolInfo
* @param    info            Info object.
* @param    name            Diagnostic name. //Not currently used for anything. Can be kNULL
* @param    id              Diagnostic id (must be unique to tool). //Not currently used for anything
* @param    diagnosticInfo  Pointer to receive the new diagnostic's info.
* @return                   Operation status.
*/
GdkFx(kStatus) GdkToolInfo_AddDiagnostic(GdkToolInfo info, const kChar* name, k32s id, GdkDiagnosticInfo* diagnosticInfo);

/**
* Returns the number of diagnostics.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Diagnostic count.
*/
GdkFx(kSize) GdkToolInfo_DiagnosticCount(GdkToolInfo info);

/**
* Clears (removes) all diagnostic outputs from tool.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkToolInfo_ClearDiagnostics(GdkToolInfo info);

/**
* Returns the diagnostic info of the diagnostic with the specified index.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    index       diagnostic index.
* @return               Diagnostic Info
*/
GdkFx(GdkDiagnosticInfo) GdkToolInfo_DiagnosticAt(GdkToolInfo info, kSize index);

/**
* Returns the id of the diagnostic with the specified index.
*
* @public               @memberof GdkToolInfo
* @param    info        Info object.
* @param    index       Diagnostic index.
* @return               Diagnostic Id
*/
GdkFx(k32s) GdkToolInfo_DiagnosticIdAt(GdkToolInfo info, kSize index);

GdkFx(kStatus) GdkToolInfo_SetIsPrivate(GdkToolInfo info, kBool isPrivate);
GdkFx(kBool) GdkToolInfo_IsPrivate(GdkToolInfo info);

GdkFx(kStatus) GdkToolInfo_EnableUnsortedSpots(GdkToolInfo info, kBool enabled);
GdkFx(kBool) GdkToolInfo_UnsortedSpotsEnabled(GdkToolInfo info);

#endif
