/**
* @file    GdkParamInfo.h
* @brief   Declares the GdkParamInfo class.
*
* Copyright (C) 2015-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GDK_EXT_PARAM_INFO_H
#define GDK_EXT_PARAM_INFO_H

#include <Gdk/GdkDef.h>

/**
* Represents the schema of a configurable parameter.
*
* @class       GdkParamInfo
* @extends     kObject
* @ingroup     Gdk-Config
*/
typedef kObject GdkParamInfo;
kDeclareClassEx(Gdk, GdkParamInfo, kObject)

/**
* @struct  GdkParamType
* @extends kValue
* @ingroup Gdk-Config
* @brief   Type of a parameter.
*/
typedef k32s GdkParamType;

/** @relates GdkParamType @{ */
#define GDK_PARAM_TYPE_UNKNOWN              (-1)        ///< Not yet set.
#define GDK_PARAM_TYPE_INT                  (0)         ///< Integer.
#define GDK_PARAM_TYPE_FLOAT                (1)         ///< Float.
#define GDK_PARAM_TYPE_BOOL                 (2)         ///< Boolean.
#define GDK_PARAM_TYPE_STRING               (3)         ///< String.
#define GDK_PARAM_TYPE_PROFILE_REGION       (4)         ///< Profile region.
#define GDK_PARAM_TYPE_SURFACE_REGION       (5)         ///< Surface region 3D.
#define GDK_PARAM_TYPE_SURFACE_REGION_2D    (6)         ///< Surface 2D region (X and Y).
#define GDK_PARAM_TYPE_GEOMETRIC_FEATURE    (7)         ///< A set of geometric features
#define GDK_PARAM_TYPE_MEASUREMENT          (8)         ///< Measurement (future)
#define GDK_PARAM_TYPE_DATA_INPUT           (9)         ///< Data Input
#define GDK_PARAM_TYPE_POINT_SET_REGION    (10)         ///< Point set region
/** @} */

/**
* Returns the parameter type.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Parameter type.
*/
GdkFx(GdkParamType) GdkParamInfo_Type(GdkParamInfo info);

/**
* Returns whether the info object is of type input data set.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               kTRUE if so, kFALSE if not.
*/
GdkFx(kBool) GdkParamInfo_IsInputData(GdkParamInfo info);

/**
* Sets the name of the parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    name        Parameter name.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_SetName(GdkParamInfo info, const kChar* name);

/**
* Returns the name of the parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Parameter name.
*/
GdkFx(const kChar*) GdkParamInfo_Name(GdkParamInfo info);

/**
* Sets a parameter to be optional or not. This is only used for backward
* compatibility with auto versioning.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    optional    Whether or not parameter is optional.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_SetConfigOptional(GdkParamInfo info, kBool optional);

/**
* Returns whether or not the parameter is optional.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Is optional.
*/
GdkFx(kBool) GdkParamInfo_ConfigOptional(GdkParamInfo info);

/**
* Sets the display label of the parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    label       Parameter label.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_SetLabel(GdkParamInfo info, const kChar* label);

/**
* Returns the display label of the parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Parameter label.
*/
GdkFx(const kChar*) GdkParamInfo_Label(GdkParamInfo info);

/**
* Sets the units of the parameter.
*
* It is optional to specify the units.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    units       Parameter units.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_SetUnits(GdkParamInfo info, const kChar* units);

/**
* Returns the units of the parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Parameter units (empty string if not set).
*/
GdkFx(const kChar*) GdkParamInfo_Units(GdkParamInfo info);

/**
* Returns the number of options for the parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Number of options.
*/
GdkFx(kSize) GdkParamInfo_OptionCount(GdkParamInfo info);

/**
* Returns the option name at a specified index.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    index       Option index.
* @return               Option name.
*/
GdkFx(const kChar*) GdkParamInfo_OptionNameAt(GdkParamInfo info, kSize index);

/**
* Remove the option at a specified index.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    index       Option index.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_RemoveOption(GdkParamInfo info, kSize index);

/**
* Clears all options for the parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_ClearOptions(GdkParamInfo info);

/**
* Sets the minimum value for an integer parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    min         Minimum value.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_SetMinInt(GdkParamInfo info, k32s min);

/**
* Returns the minimum value for an integer parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Minimum value.
*/
GdkFx(k32s) GdkParamInfo_MinInt(GdkParamInfo info);

/**
* Sets the maximum value for an integer parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    max         Maximum value.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_SetMaxInt(GdkParamInfo info, k32s max);

/**
* Returns the maximum value for an integer parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Maximum value.
*/
GdkFx(k32s) GdkParamInfo_MaxInt(GdkParamInfo info);

/**
* Add an option for an integer parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    option      Option value.
* @param    name        Option name.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_AddOptionInt(GdkParamInfo info, k32s option, const kChar* name);

/**
* Returns the option at a specified index for an integer parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    index       Option index.
* @return               Option value.
*/
GdkFx(k32s) GdkParamInfo_OptionAtInt(GdkParamInfo info, kSize index);

/**
* Sets the minimum value for a decimal parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    min         Minimum value.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_SetMinFloat(GdkParamInfo info, k64f min);

/**
* Returns the minimum value for a decimal parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Minimum value.
*/
GdkFx(k64f) GdkParamInfo_MinFloat(GdkParamInfo info);

/**
* Sets the maximum value for a decimal parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    max         Maximum value.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_SetMaxFloat(GdkParamInfo info, k64f max);

/**
* Returns the maximum value for a decimal parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Maximum value.
*/
GdkFx(k64f) GdkParamInfo_MaxFloat(GdkParamInfo info);

/**
* Add an option for an float parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    option      Option value.
* @param    name        Option name.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_AddOptionFloat(GdkParamInfo info, k64f option, const kChar* name);

/**
* Returns the option at a specified index for an float parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    index       Option index.
* @return               Option value.
*/
GdkFx(k64f) GdkParamInfo_OptionAtFloat(GdkParamInfo info, kSize index);

/**
* Assign the default value of the parameter to sue when refreshing the parameter.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    value       Pointer to default value to assign.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_SetDefault(GdkParamInfo info, const void* value);

/**
* Enables Z angle support for the surface region passed.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object - expected to be a parameter of type GDK_PARAM_TYPE_SURFACE_REGION.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_EnableSurfaceRegionZAngle(GdkParamInfo info);

/**
* If the parameter is a Multiple Data Type Input, add an accepted data type.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    type        GdkDataType to add.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_AddDataType(GdkParamInfo info, GdkDataType type);

/**
* Adds the source perspective of the data to list of source perspectives
* This is only applicable to parameters of type DataInput and will automatically
* have GDK_DATA_SOURCE_TOP added for parameters of this type.
*
* For live profile sensors with n-buddy configuration, if the source data from
* one of the buddies is required, the device's index added to GDK_DATA_SOURCE_DEVICE_BASE
* would have to be specified to be able to retrieve the corresponding data from that device.
* For example, if the source data from the second buddy in a 3 sensor buddy configuration
* is desired, add its device index of 2 to GDK_DATA_SOURCE_DEVICE_BASE = 102 to be used as
* the data source value for that device.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    source      Enumeration of data source:
*                       GDK_DATA_SOURCE_NONE
*                       GDK_DATA_SOURCE_TOP
*                       GDK_DATA_SOURCE_BOTTOM
*                       GDK_DATA_SOURCE_TOP_LEFT
*                       GDK_DATA_SOURCE_TOP_RIGHT
*                       GDK_DATA_SOURCE_TOP_BOTTOM
*                       GDK_DATA_SOURCE_LEFT_RIGHT
*                       GDK_DATA_SOURCE_DEVICE_BASE + device index
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_AddSource(GdkParamInfo info, GdkDataSource source);

/**
* Returns the list of source perspectives of the data.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               List of source perspectives set for this info object
*/
GdkFx(kArrayList) GdkParamInfo_Sources(GdkParamInfo info);

/**
* Removes the source from the list of source perspectives.
* If there are multiple matching sources, it will remove them all.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    source      Enumeration of data source:
*                       GDK_DATA_SOURCE_NONE
*                       GDK_DATA_SOURCE_TOP
*                       GDK_DATA_SOURCE_BOTTOM
*                       GDK_DATA_SOURCE_TOP_LEFT
*                       GDK_DATA_SOURCE_TOP_RIGHT
*                       GDK_DATA_SOURCE_TOP_BOTTOM
*                       GDK_DATA_SOURCE_LEFT_RIGHT
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_RemoveSource(GdkParamInfo info, GdkDataSource source);

/**
* Sets the input parameter to be allowed to be set to None (no argument selected).
* Default value is kTRUE.
* Setting to kFALSE means there must exist a valid input to which this parameter
* can be set. This parameter cannot be set to nothing, otherwise an error is generated.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    flag        Flag indicating parameter can be set to none (kTRUE) or not (kFALSE)
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_SetNoneAllowed(GdkParamInfo info, kBool flag);

/**
* Returns whether the input parameter can be set to None or requires an actual data input.
*
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               kTRUE if can be none, kFALSE if not.
*/
GdkFx(kBool) GdkParamInfo_NoneAllowed(GdkParamInfo info);


//// Deprecated API functions

/**
* [Deprecated] Use GdkParamInfo_SetConfigOptional() instead.
*
* @deprecated
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @param    optional    Whether or not parameter is optional.
* @return               Operation status.
*/
GdkFx(kStatus) GdkParamInfo_SetIsOptional(GdkParamInfo info, kBool optional);

/**
* [Deprecated] Use GdkParamInfo_ConfigOptional() instead.
*
* @deprecated
* @public               @memberof GdkParamInfo
* @param    info        Info object.
* @return               Is optional.
*/
GdkFx(kBool) GdkParamInfo_IsOptional(GdkParamInfo info);

#include <Gdk/Config/GdkParamInfo.x.h>

#endif
