/**
* @file    GdkCfgInterfaces.h
* @brief   Declares interfaces for accessing configuration implementations.
*
* Copyright (C) 2018-2022 by LMI Technologies Inc.  All rights reserved.
*/
#ifndef GDK_CFG_INTERFACES_H
#define GDK_CFG_INTERFACES_H

#include <Gdk/GdkDef.h>
#include <Gdk/Config/ExtParams.h>
#include <Gdk/Tools/GdkToolVersionInfo.h>

namespace Gdk
{
namespace Config
{

class GdkClass IToolEntity
{
public:
    virtual const kChar* TypeName() const = 0;
    virtual k32s Id() const = 0;
    virtual const kChar* Name() const = 0;
    virtual void SetName(const kChar* name) = 0;
    virtual ExtParamSet& ParamSet() = 0;

    virtual kPointer ConfigWrapper() = 0;

    virtual ~IToolEntity() {};
};

class GdkClass IToolOutput : public IToolEntity
{
public:
    virtual bool Enabled() const = 0;
    virtual void Enable(bool enabled) = 0;
    virtual GdkToolOutputType OutputType() const = 0;
};

class GdkClass IMeasurementOutput : public IToolOutput
{
public:
    virtual GdkMeasurementInfo TypeInfo() = 0;
};

class GdkClass IFeatureOutput : public IToolOutput
{
public:
    virtual GdkFeatureInfo TypeInfo() = 0;
};

class GdkClass IDataOutput : public IToolOutput
{
public:
    virtual GdkToolDataOutputInfo TypeInfo() = 0;
};

class GdkClass ITool : public IToolEntity
{
public:
    virtual GdkToolVersionInfo VersionInfo() = 0;

    virtual void SetSource(GdkDataSource source) = 0;
    virtual GdkDataSource Source() const = 0;

    virtual void SetAnchor(GdkAnchorParam axis, k32s id) = 0;
    virtual k32s Anchor(GdkAnchorParam axis) const = 0;

    virtual size_t MeasurementCount() const = 0;
    virtual IMeasurementOutput& MeasurementAt(size_t index) = 0;
    virtual IMeasurementOutput& AddMeasurement(const kChar* typeName) = 0;
    virtual void RemoveMeasurement(size_t index) = 0;

    virtual size_t FeatureCount() const = 0;
    virtual IFeatureOutput& FeatureAt(size_t index) = 0;
    virtual IFeatureOutput& AddFeature(const kChar* typeName) = 0;
    virtual void RemoveFeature(size_t index) = 0;

    virtual size_t DataCount() const = 0;
    virtual IDataOutput& DataAt(size_t index) = 0;
    virtual IDataOutput& AddData(const kChar* typeName) = 0;
    virtual void RemoveData(size_t index) = 0;

    virtual size_t OutputCount() const = 0;
    virtual IToolOutput& OutputAt(size_t index) = 0;

    virtual void* ModelPointer() const = 0;
};

}} // Namespace

#include <Gdk/GdkDef.h>

#endif 
